package com.seeq.link.sdk;

import java.time.Duration;
import java.util.Objects;

import org.apache.commons.lang.time.StopWatch;

import com.google.common.base.Preconditions;
import com.seeq.link.sdk.interfaces.Connection;
import com.seeq.link.sdk.utilities.Event;
import com.seeq.link.sdk.utilities.ThreadCollection;
import com.seeq.utilities.AutoResetEvent;
import com.seeq.utilities.BackoffRetryHelper;

import lombok.extern.slf4j.Slf4j;

/**
 * Facilitates a consistent approach external connection establishment and monitoring. External connections can be
 * connector datasources or the Seeq Server.
 *
 * Once connected, a monitoring thread will periodically call the abstract monitor() function that must be implemented
 * in derived classes.
 *
 * If the monitor function() discovers a dead connection and sets the state to Disconnected, the monitoring thread will
 * attempt reconnection at intervals that exponentially increase (to avoid flooding the log).
 *
 * See {@link Connection} for more documentation on the interface.
 */
@Slf4j
public abstract class BaseConnection implements Connection {

    @Override
    public abstract void initialize();

    @Override
    public abstract void destroy();

    private volatile ConnectionState state = ConnectionState.DISABLED;

    @Override
    public ConnectionState getState() {
        return this.state;
    }

    private String connectionMessage = "";

    @Override
    public String getConnectionMessage() {
        return this.connectionMessage;
    }

    protected Exception lastException;

    /**
     * The last exception that has occurred on the connection.
     *
     * @return the last exception
     */
    public Exception getLastException() {
        return this.lastException;
    }

    private final Event<StateChangedEventArgs> stateChangedEvent = new Event<>();

    @Override
    public Event<StateChangedEventArgs> getStateChangedEvent() {
        return this.stateChangedEvent;
    }

    /**
     * The connection ID used to connect to the server.
     *
     * @return the connection ID
     */
    protected abstract String getConnectionId();

    private final Object setStateLock = new Object();

    /**
     * Set the current state of the connection. If the new state is Disconnected and automaticallyReconnect is true, an
     * attempt will be made to reconnect within the ReconnectDelay.
     *
     * @param newState
     *         The new state for the connection.
     * @param message
     *         A message that indicates the status of the connection.
     */
    protected void setState(ConnectionState newState, String message) {
        synchronized (this.setStateLock) {
            if (this.state == newState && Objects.equals(this.connectionMessage, message)) {
                return;
            }

            if (this.state == ConnectionState.DISABLED) {
                // If a Connected or Disconnected event comes in on a thread after we've disabled the connection,
                // ignore it -- the connection is disabled. The connection will have to go through the Connecting
                // state to be enabled properly
                if (newState == ConnectionState.CONNECTED || newState == ConnectionState.DISCONNECTED) {
                    return;
                }
            }

            this.state = newState;
            this.connectionMessage = message;

            LOG.info("{} connection state changed to {}. {}",
                    this.getConnectionId(), newState, message);

            this.stateChangedEvent.dispatch(this, new StateChangedEventArgs(this, newState, message));
        }
    }

    private final AutoResetEvent connectionMonitorWakeup = new AutoResetEvent(false);

    /**
     * This function runs on a worker thread and facilitates monitoring of a connection for vitality. It calls the
     * monitor() function periodically and if the connection becomes disconnected, this thread will call connect()
     * periodically to try to reestablish a connection.
     *
     * The thread will be shutdown when Disable() is called.
     */
    private void connectionMonitor() {
        try {
            Thread.currentThread().setName(String.format("Connection: %s", this.getConnectionId()));

            while (true) {
                if (this.getState().equals(ConnectionState.DISCONNECTED)) {
                    try {
                        this.connect();
                    } catch (Exception e) {
                        this.handleConnectionMonitorException("connect()", e);
                    }
                }

                if (!this.getState().equals(ConnectionState.DISCONNECTED)) {
                    try {
                        this.monitor();
                    } catch (Exception e) {
                        this.handleConnectionMonitorException("monitor()",  e);
                    }
                }

                if (this.getState().equals(ConnectionState.DISCONNECTED)) {
                    LOG.info("Waiting {} seconds to attempt reconnect for {}", this.currentReconnectDelay.getSeconds(),
                            this.getConnectionId());
                    this.connectionMonitorWakeup.waitOne(this.currentReconnectDelay);

                    // Exponential backoff to avoid flooding the log file
                    this.retryHelper.advance();
                    this.currentReconnectDelay = Duration.ofMillis((long) this.retryHelper.getWaitInterval());
                    this.boundCurrentReconnectDelay();
                } else {
                    this.connectionMonitorWakeup.waitOne(this.monitorPeriod);

                    // Set reconnect delay back to the minimum in case it was altered.
                    this.currentReconnectDelay = this.minReconnectDelay;
                    this.retryHelper.reset();
                }
            }
        } catch (InterruptedException e) {
            return;
        }
    }

    protected void handleConnectionMonitorException(String methodName, Exception exception) {
        LOG.error("Exception thrown by {} method:", methodName, exception);
    }

    private Duration currentReconnectDelay = Duration.ofSeconds(5);
    private Duration maxReconnectDelay = Duration.ofMinutes(1);
    private Duration minReconnectDelay = Duration.ofSeconds(5);

    /**
     * Time to wait, default is 5 seconds. Connectors use an exponential backoff algorithm to reduce the noise in the
     * log files when a connection is down.
     */
    @Override
    public Duration getMinReconnectDelay() {
        return this.minReconnectDelay;
    }

    @Override
    public void setMinReconnectDelay(Duration value) {
        Preconditions.checkArgument(value.toMillis() > 0, "MinReconnectDelay cannot be zero");

        this.minReconnectDelay = value;
        this.boundCurrentReconnectDelay();
        this.initBackoffRetryHelper();
    }

    @Override
    public Duration getMaxReconnectDelay() {
        return this.maxReconnectDelay;
    }

    @Override
    public void setMaxReconnectDelay(Duration value) {
        Preconditions.checkArgument(value.toMillis() > 0, "MaxReconnectDelay cannot be zero");

        this.maxReconnectDelay = value;
        this.boundCurrentReconnectDelay();
        this.initBackoffRetryHelper();
    }

    private void boundCurrentReconnectDelay() {
        if (this.currentReconnectDelay.compareTo(this.maxReconnectDelay) > 0) {
            this.currentReconnectDelay = this.maxReconnectDelay;
        }
        if (this.currentReconnectDelay.compareTo(this.minReconnectDelay) < 0) {
            this.currentReconnectDelay = this.minReconnectDelay;
        }
    }

    private BackoffRetryHelper retryHelper = new BackoffRetryHelper(this.minReconnectDelay.toMillis(),
            this.maxReconnectDelay.toMillis(), new StopWatch());

    private void initBackoffRetryHelper() {
        this.retryHelper = new BackoffRetryHelper(this.minReconnectDelay.toMillis(),
                this.maxReconnectDelay.toMillis(), new StopWatch());
    }

    private Duration monitorPeriod = Duration.ofSeconds(5);

    @Override
    public Duration getMonitorPeriod() {
        return this.monitorPeriod;
    }

    @Override
    public void setMonitorPeriod(Duration value) {
        Preconditions.checkArgument(value.toMillis() > 0, "MonitorPeriod cannot be zero");

        this.monitorPeriod = value;
    }

    private final ThreadCollection backgroundThreads = new ThreadCollection(this.getClass().getName());

    public ThreadCollection getBackgroundThreads() {
        return this.backgroundThreads;
    }

    /**
     * Enable connections to the server and attempt to make a connection. This calls the connect method in the derived
     * class.
     */
    @Override
    public void enable() {
        if (this.getState() != ConnectionState.DISABLED) {
            LOG.warn("{} connection already enabled", this.getConnectionId());
            return;
        }
        LOG.info("{} connection enabled", this.getConnectionId());

        // Set state to Disconnected here (without causing an event to be fired)
        // so that connection thread will attempt to connect immediately.
        this.state = ConnectionState.DISCONNECTED;

        this.backgroundThreads.setID(this.getConnectionId());
        this.backgroundThreads.spawn(this::connectionMonitor);
    }

    /**
     * Disable connections to the server. This calls the disconnect method in the derived class.
     */
    @Override
    public void disable() {
        if (this.getState() == ConnectionState.DISABLED) {
            LOG.warn("{} connection already disabled", this.getConnectionId());
        } else {
            // We disable the connection first before disconnecting because there is a background
            // thread running connectionMonitor() regularly, and if it finds state DISCONNECTED it
            // will reconnect us.
            LOG.info("{} connection disabling", this.getConnectionId());
            this.setState(ConnectionState.DISABLED, "");
            LOG.info("{} connection disabled", this.getConnectionId());

            // This method will try to update the state to DISCONNECTED but it will be ignored since
            // it is already DISABLED.
            LOG.info("{} disconnecting", this.getConnectionId());
            this.disconnect();
        }

        this.backgroundThreads.shutDownAll();
    }

    /**
     * Connect to the server (implemented by derived classes).
     */
    protected abstract void connect();

    /**
     * A connector-specific test to ensure the connection is alive (implemented by derived classes).
     */
    protected abstract void monitor();

    /**
     * Disconnect from the server (implemented by derived classes).
     */
    protected abstract void disconnect();
}
