package com.sdk.wisetracker.base.tracker.common.log;

import android.annotation.SuppressLint;
import android.text.TextUtils;
import android.util.Log;

import com.google.gson.JsonObject;
import com.sdk.wisetracker.base.tracker.data.manager.SessionDataManager;
import com.sdk.wisetracker.base.tracker.data.model.Session;
import com.sdk.wisetracker.base.tracker.network.SessionApi;
import com.sdk.wisetracker.base.tracker.network.SessionRetrofit;
import com.sdk.wisetracker.base.tracker.network.request.RequestError;
import com.sdk.wisetracker.base.tracker.network.type.HeaderType;

import java.util.ArrayList;
import java.util.List;

import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.functions.Consumer;
import io.reactivex.schedulers.Schedulers;

/**
 * SDK 발생 로그 관리 유틸 클래스
 */
public class WiseLog {

    // private static final Pattern classPattern = Pattern.compile("(\\$\\d+)+$");
    private static final String TAG = "[WiseLog]";
    private static List<String> logList = new ArrayList<>();
    private static boolean isDebug = true;

    public static void d(String message) {
        message = getMessage(Log.DEBUG, message);
        debugPrint(message);
    }

    private static void debugPrint(String message) {
        if (isDebug) {
            Log.d(TAG, message);
        } else {
            logList.add(message);
        }
    }

    public static void e(Throwable throwable) {
        String message = getMessage(Log.ERROR, null);
        String errorMessage = getErrorMessage(throwable);
        errorPrint(message, errorMessage);
    }

    public static void e(String message, Throwable throwable) {
        message = getMessage(Log.ERROR, message);
        String errorMessage = getErrorMessage(throwable);
        errorPrint(message, errorMessage);
    }

    private static void errorPrint(String message, String errorMessage) {
        if (isDebug) {
            Log.e(TAG, message);
            Log.e(TAG, errorMessage);
        } else {
            StringBuilder allLog = new StringBuilder();
            allLog.append("---- ERROR PRINT----\n");
            for (String log : logList) {
                allLog.append(log).append("\n");
            }
            allLog.append(message + "\n");
            allLog.append(errorMessage + "\n");
            Log.e(TAG, allLog.toString());
            sendLogData(allLog.toString());
        }
    }

    private static String getCallInfo(int index) {
        StackTraceElement[] stackTraceElements = new Throwable().getStackTrace();
        StackTraceElement element = stackTraceElements[index];
        String className = element.getClassName();
        String classArray[] = className.split("\\.");
        className = classArray[classArray.length - 1];
        String methodName = element.getMethodName();
        int lineNumber = element.getLineNumber();
        return className + "#" + methodName + "(" + lineNumber + ")";
    }

    private static String getMessage(int priority, String message) {
        String status = priority == Log.DEBUG ? "#DEBUG" : "#ERROR";
        String info = getCallInfo(3);
        if (TextUtils.isEmpty(message)) {
            return info + status + "_CALL";
        } else {
            return info + status + "_MESSAGE#" + message;
        }
    }

    private static String getErrorMessage(Throwable throwable) {
        String errorMessage = "#EXCEPTION_MESSAGE#" + throwable.getMessage();
        return errorMessage;
    }

    private static RequestError getRequestError(String logString) {
        try {
            Session session = SessionDataManager.getInstance().getSession();
            RequestError requestError = new RequestError();
            requestError.setServiceNumber(String.valueOf(session.getServiceNumber()));
            requestError.setErrLog(logString);
            requestError.setDeepLink(session.getDeepLinkReferrer());
            requestError.setInstallReferrer(session.getInstallReferrer());
            return requestError;
        } catch (Exception e) {
            Log.d(TAG, "#CREATE_REQUEST_ERROR_FAIL");
        }
        return new RequestError();
    }

    // 예외 발생시 collector 로그 데이터 전송
    @SuppressLint("CheckResult")
    private static void sendLogData(String logDataString) {
        SessionApi api = SessionRetrofit.getInstance().getBaseApi(SessionApi.class, HeaderType.DOT);
        api.sendLog(getRequestError(logDataString))
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Consumer<JsonObject>() {
                    @Override
                    public void accept(JsonObject jsonObject) {
                        Log.d(TAG, "#SEND_LOG_DATA_SUCCESS");
                        if (jsonObject != null) {
                            Log.d(TAG, "#JSON_STRING ->" + jsonObject.toString());
                        }
                        logList.clear();
                    }
                }, new Consumer<Throwable>() {
                    @Override
                    public void accept(Throwable throwable) {
                        Log.d(TAG, "#SEND_LOG_DATA_FAIL");
                        Log.d(TAG, "#" + throwable.getMessage());
                    }
                });
    }

}