package com.sdk.wisetracker.base.tracker.data.manager;

import android.annotation.SuppressLint;
import android.net.Uri;
import android.os.Build;
import android.text.TextUtils;

import com.google.gson.Gson;
import com.sdk.wisetracker.base.tracker.common.log.WiseLog;
import com.sdk.wisetracker.base.tracker.data.init.ReferrerInitCallBack;
import com.sdk.wisetracker.base.tracker.data.model.Priority;
import com.sdk.wisetracker.base.tracker.data.model.Referrer;
import com.sdk.wisetracker.base.tracker.data.model.Session;
import com.sdk.wisetracker.base.tracker.network.SessionApi;
import com.sdk.wisetracker.base.tracker.network.SessionRetrofit;
import com.sdk.wisetracker.base.tracker.network.request.RequestFingerPrint;
import com.sdk.wisetracker.base.tracker.network.type.HeaderType;
import com.sdk.wisetracker.base.tracker.util.TimeUtil;

import io.reactivex.Observable;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.functions.Consumer;
import io.reactivex.schedulers.Schedulers;

/**
 * 핑거프린트 처리 class
 */
public class FingerPrintDataManager {

    private static FingerPrintDataManager instance = null;
    private Session session;

    public static FingerPrintDataManager getInstance() {
        if (instance == null) {
            instance = new FingerPrintDataManager();
        }
        return instance;
    }

    public FingerPrintDataManager() {
        session = SessionDataManager.getInstance().getSession();
    }

    /**
     * 핑거프린트 API 호출 Request 정보 설정
     * @return
     */
    private RequestFingerPrint getRequestFingerPrint() {
        try {
            RequestFingerPrint requestFingerPrint = new RequestFingerPrint();
            requestFingerPrint.setServiceNumber(String.valueOf(session.getServiceNumber()));
            requestFingerPrint.setAdId(session.getAdId());
            requestFingerPrint.setDeviceModel(Build.MODEL);
            requestFingerPrint.setOs(String.valueOf(Build.VERSION.SDK_INT));
            requestFingerPrint.setNew(BasicDataManager.getInstance().getBasicData().getIsFirstAppOpen().equals("Y"));
            requestFingerPrint.setClickTime(session.getAdClickTime());
            return requestFingerPrint;
        } catch (Exception e) {
            WiseLog.e(e);
        }
        return null;
    }

    private Observable<Referrer> getFingerPrintObservable() {
        SessionApi api = SessionRetrofit.getInstance().getBaseApi(SessionApi.class, HeaderType.FRP);
        return api.sendFingerPrint(getRequestFingerPrint())
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread());
    }

    /**
     * 앱 설치 후 최초 실행시 핑거프린트 호출
     * @param callBack
     */
    @SuppressLint("CheckResult")
    public void loadFingerPrintByInstall(ReferrerInitCallBack callBack) {

        try {

            getFingerPrintObservable()
                    .subscribe(new Consumer<Referrer>() {
                        @Override
                        public void accept(Referrer fingerPrint) {
                            if (fingerPrint != null) {
                                WiseLog.d("FingerPrint : " + new Gson().toJson(fingerPrint));
                            }
                            //String fp = "{\"wtaffid\":\"\",\"_adset\":\"\",\"_ad\":\"\",\"wtufn\":\"direct\",\"_partner\":\"\",\"attributionPriorities\":{\"ATR001\":0,\"ATR002\":0,\"ATR003\":0,\"ATR004\":0,\"ATRSAD\":0,\"ATRLNKID\":0},\"wtbffid\":\"\",\"wtpid\":\"\",\"etd4\":\"\",\"etd3\":\"\",\"_adtype\":\"\",\"etd2\":\"\",\"etd1\":\"\",\"_adId\":\"\",\"wtref\":\"wts=FACEBOOK&wtc=C1577074799542&wtm=&wtw=&wtaffid=&wtbffid=&wtclkTime=1585895277047&wtp=1&_matNo=M15846086641060000102&_wtcid=testwtcid&_etd1=&_etd2=&_etd3=&_etd4=&_etd5=&_etd6=&_etd7=&_etd8=&_etd9=&_etd10=&_partner=&_adset=&_adsetId=&_ad=&_adId=&_adtype=&_wtpid=&_wcost=0.0&_wevtnm=&_wcurcy=KRW&_isRetarget=Y&_wtufn=direct&_wtckp=144000&attriType=ATR002\",\"wevtnm\":\"\",\"isRetarget\":\"Y\",\"itp\":144000,\"wtclkTime\":1585895277047,\"wcurcy\":\"KRW\",\"responseTime\":1585711189527,\"wtc\":\"C1577074799542\",\"etd9\":\"\",\"matNo\":\"M15846086641060000102\",\"etd8\":\"\",\"etd7\":\"\",\"etd6\":\"\",\"etd5\":\"\",\"_adsetId\":\"\",\"wtm\":\"\",\"etd10\":\"\",\"wtp\":1,\"wts\":\"FACEBOOK\",\"wtw\":\"\",\"wcost\":0.0}";
                            //fingerPrint = new Gson().fromJson(fp, Referrer.class);
                            if (!isValidateFingerPrint(fingerPrint)) {
                                // 핑거프린트 유효성 체크 fail
                                callBack.onCallback(true);
                                return;
                            }
                            if (isEmptyReferrer(fingerPrint)) {
                                callBack.onCallback(true);
                                return;
                            }
                            checkInstallReferrerPriority(fingerPrint);
                            callBack.onCallback(true);
                        }
                    }, new Consumer<Throwable>() {
                        @Override
                        public void accept(Throwable throwable) {
                            callBack.onCallback(true);
                            WiseLog.e("load finger print error !", throwable);
                        }
                    });

        } catch (Exception e) {
            WiseLog.e("load finger print error !", e);
        }

    }

    /**
     * 딥링크 진입시 핑거프린트 호출
     */
    @SuppressLint("CheckResult")
    public void loadFingerPrintByDeepLink() {

        try {

            getFingerPrintObservable()
                    .subscribe(new Consumer<Referrer>() {
                        @Override
                        public void accept(Referrer referrer) {
                            if (!isValidateFingerPrint(referrer)) {
                                // 레퍼러 유효성 체크 fail
                                return;
                            }
                            setDeepLink(referrer);
                        }
                    }, new Consumer<Throwable>() {
                        @Override
                        public void accept(Throwable throwable) {
                            WiseLog.e("load finger print error !", throwable);
                        }
                    });

        } catch (Exception e) {
            WiseLog.e(e);
        }

    }

    /**
     * 핑거 프린트 유효성 체크
     * @param fingerPrint
     * @return 유효한 핑거프린트 일 경우 true
     */
    private boolean isValidateFingerPrint(Referrer fingerPrint) {

        try {

            if (fingerPrint == null) {
                return false;
            }

            BasicData basicData = BasicDataManager.getInstance().getBasicData();
            long lastFingerPrintTime = basicData.getLastFingerPrintTime();
            long currentFingerPrintClickTime = fingerPrint.getAdClickTime();
            WiseLog.d("last finger print time : " + lastFingerPrintTime);
            WiseLog.d("current finger print click time : " + currentFingerPrintClickTime);

            if (lastFingerPrintTime != 0 && currentFingerPrintClickTime < lastFingerPrintTime) {
                return false;
            }
            basicData.setLastFingerPrintTime(System.currentTimeMillis());
            BasicDataManager.getInstance().saveBasicData(basicData);
            return true;

        } catch (Exception e) {
            WiseLog.e("check validate finger print error !", e);
        }

        return false;

    }

    /**
     * 저장된 레퍼러 정보 확인
     * @param fingerPrint
     * @return 레퍼러 정보 유무에 따라 return
     */
    private boolean isEmptyReferrer(Referrer fingerPrint) {
        String installSource = session.getAdInstallSource();
        String installCampaign = session.getAdInstallCampaign();
        if (TextUtils.isEmpty(installSource) || TextUtils.isEmpty(installCampaign)) {
            setInstallReferrer(fingerPrint);
            return true;
        }
        return false;
    }

    /**
     * 응답 받은 핑거 프린트 값 우선 순위 설정
     * @param fingerPrint
     */
    private void checkInstallReferrerPriority(Referrer fingerPrint) {

        try {

            Priority priority = fingerPrint.getPriority();
            if (priority == null) {
                return;
            }

            String attributeType;
            int referrerPriority;
            if (TextUtils.isEmpty(session.getClickId())) {
                // click id 미존재
                attributeType = "ATR003";
                referrerPriority = priority.getReferrerPriority();
            } else {
                // click id 존재
                attributeType = "ATR001";
                referrerPriority = priority.getClickPriority();
                session.setClickId(null);
            }

            String link = fingerPrint.getAdLinkUrl();
            if (TextUtils.isEmpty(link)) {
                return;
            }

            Uri uri = Uri.parse("?" + link);
            String fingerPrintType = uri.getQueryParameter("attriType");
            String replaceType;
            if (TextUtils.isEmpty(fingerPrintType)) {
                fingerPrintType = attributeType;
                replaceType = "attriType=";
            } else {
                replaceType = "attriType=" + fingerPrintType;
            }

            int fingerPrintPriority;
            if (fingerPrintType.equals("ATR002")) {
                fingerPrintPriority = priority.getAdvertisingPriority();
            } else {
                fingerPrintPriority = priority.getFingerPrintPriority();
            }

            if (referrerPriority > fingerPrintPriority) {
                // 핑거프린트 우선순위 높음
                setInstallReferrer(fingerPrint);
            } else {
                String changeType = "attriType=" + attributeType;
                link = link.replace(replaceType, changeType);
                fingerPrint.setAdLinkUrl(link);
                session.setAdLinkUrl(link);
                SessionDataManager.getInstance().saveSession(session);
            }

        } catch (Exception e) {
            WiseLog.e("check finger print priority error !!", e);
        }

    }

    /**
     * 딥링크 정보 유효성 체크
     * @param deepLink
     * @return 딥링크 유효성에 따른 return
     */
    private boolean isValidDeepLink(Referrer deepLink) {
        long clickTime = deepLink.getAdClickTime();
        int installExpireTime = deepLink.getAdExpireDay(); // wtp
        if (clickTime + TimeUtil.DAY_MILLIS * installExpireTime < System.currentTimeMillis()) {
            WiseLog.d("invalid deep link fp");
            return false;
        }
        if (TextUtils.isEmpty(session.getAdCampaign()) || TextUtils.isEmpty(session.getAdSource())) {
            WiseLog.d("invalid data");
            return false;
        }
        if (!TextUtils.isEmpty(session.getAdSource()) && !session.getAdSource().equals(deepLink.getAdSource())) {
            WiseLog.d("invalid data");
            return false;
        }
        if (!TextUtils.isEmpty(session.getAdCampaign()) && !session.getAdCampaign().equals(deepLink.getAdCampaign())) {
            WiseLog.d("invalid data");
            return false;
        }
        if (session.getAdClickTime() == deepLink.getAdClickTime()) {
            WiseLog.d("invalid data");
            return false;
        }
        return true;
    }

    /**
     * 딥링크 정보 저장
     * @param deepLink
     */
    private void setDeepLink(Referrer deepLink) {
        if (!isValidDeepLink(deepLink)) {
            return;
        }
        WiseLog.d("set deep link by finger print");
        session.setAdSource(deepLink.getAdSource());
        session.setAdType(deepLink.getAdType());
        session.setAdCampaign(deepLink.getAdCampaign());
        session.setAdKeyword(deepLink.getAdKeyword());
        session.setAdClickTime(deepLink.getAdClickTime());
        session.setAdExpireDay(deepLink.getAdExpireDay());
        session.setDepth1(deepLink.getDepth1());
        session.setDepth2(deepLink.getDepth2());
        session.setAdLinkUrl(deepLink.getAdLinkUrl());
        session.setIsDeepLinkUpdate("Y");
        session.setGoogleCampaignId(deepLink.getGoogleCampaignId());
        // session.setReferrerType("FINGER_PRINT");
        SessionDataManager.getInstance().saveSession(session);
        SessionDataManager.deepLinkSubject.onNext(true);
    }

    /**
     * 인스톨 레퍼러 유효성 체크
     * @param installReferrer
     * @return 인스톨 레퍼러 유효성에 따라 return
     */
    private boolean isValidInstallReferrer(Referrer installReferrer) {
        long clickTime = installReferrer.getAdClickTime();
        int installExpireTime = installReferrer.getAdInstallExpireTime(); // itp
        if (clickTime + (TimeUtil.MINUTE_MILLIS * installExpireTime) < System.currentTimeMillis()) {
            WiseLog.d("invalid install fp ");
            return false;
        }
        if (TextUtils.isEmpty(installReferrer.getAdSource()) || TextUtils.isEmpty(installReferrer.getAdCampaign())) {
            WiseLog.d("invalid install fp data");
            return false;
        }
        return true;
    }

    /**
     * 인스톨 레퍼러 정보 저장
     * @param installReferrer
     */
    private void setInstallReferrer(Referrer installReferrer) {
        if (!isValidInstallReferrer(installReferrer)) {
            return;
        }
        WiseLog.d("set install referrer by finger print");
        session.setAdInstallSource(installReferrer.getAdSource());
        session.setAdInstallType(installReferrer.getAdType());
        session.setAdInstallCampaign(installReferrer.getAdCampaign());
        session.setAdInstallKeyword(installReferrer.getAdKeyword());
        session.setAdInstallClickTime(installReferrer.getAdClickTime());
        session.setAdInstallExpireTime(installReferrer.getAdInstallExpireTime());
        session.setInstallDepth1(installReferrer.getDepth1());
        session.setInstallDepth2(installReferrer.getDepth2());
        session.setInstallReferrer(installReferrer.getAdLinkUrl());
        session.setIsReferrerUpdate("Y");
        session.setGoogleCampaignId(installReferrer.getGoogleCampaignId());
        if (session.getAdExpireDay() == 0) {
            // set wtp
            session.setAdExpireDay(installReferrer.getAdExpireDay());
        }
        SessionDataManager.getInstance().saveSession(session);
    }

}
