package com.sdk.wisetracker.base.tracker.data.manager;

import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.text.TextUtils;

import com.sdk.wisetracker.base.tracker.common.log.WiseLog;
import com.sdk.wisetracker.base.tracker.data.model.Const;
import com.sdk.wisetracker.base.tracker.util.TimeUtil;
import com.sdk.wisetracker.base.tracker.util.ValueOf;

import java.util.Iterator;
import java.util.Set;

/**
 * 딥링크 처리 class
 */
public class DeepLinkDataManager extends BaseDataManager {

    private static DeepLinkDataManager instance = null;

    public static DeepLinkDataManager getInstance() {
        if (instance == null) {
            instance = new DeepLinkDataManager();
        }
        return instance;
    }

    @Override
    public void setData() {
        checkValidation();
    }

    @Override
    public void afterTransaction() {
        session.setOpenDeepLink(null);
        session.setDeepLinkReferrer(null);
        super.afterTransaction();
    }

    /**
     * 딥링크 유효성 체크
     */
    private void checkValidation() {

        try {

            String source  = session.getAdSource();
            String campaign = session.getAdCampaign();
            if(TextUtils.isEmpty(source) || TextUtils.isEmpty(campaign)) {
                return;
            }

            long adClickTime = session.getAdClickTime();
            int adExpireDay = session.getAdExpireDay();
            if (adClickTime == 0 || adExpireDay == 0) {
                resetDeepLink();
                return;
            }

            long expireTime = adClickTime + (TimeUtil.DAY_MILLIS * adExpireDay);
            if (expireTime < System.currentTimeMillis()) {
                resetDeepLink();
                return;
            }

        } catch (Exception e) {
            WiseLog.e(e);
        }

    }

    /**
     * 딥링크 출처 획득 후 딥링크 저장
     * @param intent
     */
    public void openDeepLink(Intent intent) {

        try {

            if (intent == null || intent.getData() == null || TextUtils.isEmpty(intent.getData().toString())) {
                return;
            }

            Uri uri = intent.getData();
            Bundle bundle = intent.getExtras();

            if (bundle != null && bundle.containsKey(Const.DEEPLINK_REFERRER)) {
                Uri referrer = (Uri) bundle.get(Const.DEEPLINK_REFERRER);
                if (referrer != null) {
                    setDeepLink(uri.toString(), referrer.toString());
                    return;
                }
            }

            String referrerName = intent.getStringExtra(Const.DEEPLINK_REFERRER_NAME);
            if (!TextUtils.isEmpty(referrerName)) {
                setDeepLink(uri.toString(), Uri.parse(referrerName).toString());
                return;
            }

            setDeepLink(uri.toString(), null);

        } catch (Exception e) {
            WiseLog.e("open deep link error", e);
        }

    }

    /**
     * 딥링크 출처 획득 후 딥링크 저장
     * @param deepLinkUrl
     */
    public void openDeepLink(String deepLinkUrl) {
        setDeepLink(deepLinkUrl, null);
    }

    /**
     * 딥링크 정보 저장
     * @param deepLinkUri
     * @param openReferrer
     */
    private void setDeepLink(String deepLinkUri, String openReferrer) {

        session.setOpenDeepLink(deepLinkUri);

        WiseLog.d("set deep link data");
        WiseLog.d("deep link url : " + deepLinkUri);
        WiseLog.d("open referrer : " + (!TextUtils.isEmpty(openReferrer) ? openReferrer : "empty"));

        if (deepLinkUri.contains("wtc") && deepLinkUri.contains("wts")) {

            // 저장 데이터 초기화
            resetDeepLink();

            Uri uri = Uri.parse(deepLinkUri);
            Set<String> queryParameters = uri.getQueryParameterNames();
            Iterator<String> iterator = queryParameters.iterator();
            while (iterator.hasNext()) {

                String name = iterator.next();
                switch (name) {
                    case Const.WTS:
                        session.setAdSource(uri.getQueryParameter(name));
                        break;
                    case Const.WTM:
                        session.setAdType(uri.getQueryParameter(name));
                        break;
                    case Const.WTC:
                        session.setAdCampaign(uri.getQueryParameter(name));
                        break;
                    case Const.WTW:
                        session.setAdKeyword(uri.getQueryParameter(name));
                        break;
                    case Const.WTCLKTIME:
                        session.setAdClickTime(ValueOf.longValue(uri.getQueryParameter(name)));
                        break;
                    case Const.WTP:
                        session.setAdExpireDay(ValueOf.intValue(uri.getQueryParameter(name)));
                        break;
                    case Const.WTAFFID:
                        session.setDepth1(uri.getQueryParameter(name));
                        break;
                    case Const.WTBFFID:
                        session.setDepth2(uri.getQueryParameter(name));
                        break;
                    case Const.WGCMPID:
                        session.setGoogleCampaignId(uri.getQueryParameter(name));
                        break;
                }

            }

            if (TextUtils.isEmpty(session.getAdSource()) || TextUtils.isEmpty(session.getAdCampaign())) {
                WiseLog.d("deep link doesn't have primary key");
                resetDeepLink();
                return;
            }

            if (session.getAdClickTime() == 0) {
                session.setAdClickTime(System.currentTimeMillis());
            }

            long expireTime = session.getAdClickTime() + (TimeUtil.DAY_MILLIS * Integer.valueOf(session.getAdExpireDay()));
            if (expireTime < System.currentTimeMillis()) {
                WiseLog.d("invalid deep link referrer");
                resetDeepLink();
                return;
            }

            session.setIsDeepLinkUpdate("Y");
            session.setAdLinkUrl(deepLinkUri);
            session.setDeepLinkReferrer(openReferrer);
            SessionDataManager.deepLinkSubject.onNext(true);

        } else {

            // finger print call
            FingerPrintDataManager.getInstance().loadFingerPrintByDeepLink();

        }

    }

    private void resetDeepLink() {
        session.setAdSource(null);
        session.setAdType(null);
        session.setAdCampaign(null);
        session.setAdKeyword(null);
        session.setAdClickTime(0);
        session.setAdExpireDay(0);
        session.setDepth1(null);
        session.setDepth2(null);
        session.setGoogleCampaignId(null);
    }

}
