package com.sdk.wisetracker.base.tracker.data.init;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.location.LocationManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Build;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.util.DisplayMetrics;
import android.view.WindowManager;

import com.sdk.wisetracker.base.BuildConfig;
import com.sdk.wisetracker.base.tracker.common.log.WiseLog;
import com.sdk.wisetracker.base.tracker.data.manager.SessionDataManager;
import com.sdk.wisetracker.base.tracker.data.model.Session;

import java.lang.reflect.Field;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

/**
 * 실행 단말기 정보 추출 및 저장
 */
public class InitEnvironment {

    private static InitEnvironment instance = null;
    private Context context;
    private Session session;

    public static InitEnvironment getInstance() {
        if (instance == null) {
            instance = new InitEnvironment();
        }
        return instance;
    }

    public InitEnvironment() {
        session = SessionDataManager.getInstance().getSession();
    }

    public void setEnvironment(Context context) {
        this.context = context;
        setLanguage();
        setCountry();
        setTimeZone();
        setOs();
        setResolution();
        setDeviceModel();
        setPackageInfo();
        setGps();
        setCarriers();
        setNetworkInfo();
        setInch();
        setUuid();
        setDeviceUuid();
        setVendorId();
        setSDKVersion();
        saveSession();
    }

    private void saveSession() {
        SessionDataManager.getInstance().saveSession(session);
    }

    private void setLanguage() {
        Locale locale = context.getResources().getConfiguration().locale;
        session.setLanguage(locale.getLanguage());
    }

    private void setCountry() {
        Locale locale = context.getResources().getConfiguration().locale;
        String country = locale.getCountry();
        TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(context.TELEPHONY_SERVICE);
        if (telephonyManager != null && !TextUtils.isEmpty(telephonyManager.getSimCountryIso())) {
            country = telephonyManager.getSimCountryIso();
        }
        session.setCountry(country.toUpperCase());
    }

    private void setTimeZone() {
        Calendar mCalendar = new GregorianCalendar();
        TimeZone mTimeZone = mCalendar.getTimeZone();
        int mGMTOffset = mTimeZone.getRawOffset();
        session.setTimeZone(String.valueOf(TimeUnit.HOURS.convert(mGMTOffset, TimeUnit.MILLISECONDS)));
    }

    private void setOs() {
        try {
            Field[] fields = Build.VERSION_CODES.class.getFields();
            String osName = fields[Build.VERSION.SDK_INT + 1].getName();
            session.setOs("Android " + osName + " (" + String.valueOf(Build.VERSION.SDK_INT) + ")");
        } catch (Exception e) {
            session.setOs("Android " + " (" + String.valueOf(Build.VERSION.SDK_INT) + ")");
        }
    }

    private void setResolution() {
        DisplayMetrics displayMetrics = new DisplayMetrics();
        WindowManager wm = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        wm.getDefaultDisplay().getMetrics(displayMetrics);
        int deviceDensityDIP = displayMetrics.densityDpi;
        int width = displayMetrics.widthPixels;
        int height = displayMetrics.heightPixels;
        String resolution = String.valueOf(width) + "*" + String.valueOf(height);
        /*
        if (!(String.valueOf(deviceDensityDIP)).equals("")) {
            resolution += ("/" + String.valueOf(deviceDensityDIP));
        }*/
        session.setResolution(resolution);
    }

    private void setDeviceModel() {
        session.setDeviceModel(Build.MODEL);
    }

    private void setPackageInfo() {
        try {
            PackageInfo packageInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
            session.setAppVersion(packageInfo.versionName);
            session.setAppInstallTime(packageInfo.firstInstallTime);
            session.setAppUpdateTime(packageInfo.lastUpdateTime);
            session.setPackageName(packageInfo.packageName);
        } catch (PackageManager.NameNotFoundException e) {
            WiseLog.d("set package name not found exception !! " + e.getMessage());
        } catch (Exception e) {
            WiseLog.d("set app version error !! " + e.getMessage());
        }
    }

    private void setGps() {
        boolean isGps;
        try {
            LocationManager locationManager = (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);
            if (locationManager == null) {
                isGps = false;
            } else {
                isGps = locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER);
            }
            session.setGps(isGps ? "Y" : "N");
        } catch (SecurityException e) {
            session.setGps("N");
            WiseLog.d("set gps security exception error !! " + e.getMessage());
        } catch (Exception e) {
            session.setGps("N");
            WiseLog.d("set gps exception error !! " + e.getMessage());
        }
    }

    private void setCarriers() {
        try {
            TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(context.TELEPHONY_SERVICE);
            if (telephonyManager == null) {
                return;
            }
            String carriers = telephonyManager.getSimOperatorName();
            session.setCarriers(carriers);
        } catch (Exception e) {
            WiseLog.d("set carriers error !! " + e.getMessage());
        }
    }

    private boolean checkSelfPermission(String permission) {
        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                return context.checkSelfPermission(permission) == PackageManager.PERMISSION_GRANTED;
            }
            return true;
        } catch (Exception e) {
            WiseLog.e("check : " + permission + "self permission error !!", e);
        }
        return false;
    }

    private void setNetworkInfo() {
        if (!checkSelfPermission(Manifest.permission.ACCESS_NETWORK_STATE)) {
            return;
        }
        try {
            int isWifi = 0;
            ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
            NetworkInfo info = connectivityManager.getActiveNetworkInfo();
            if (info != null && info.isConnected()) {
                if (info.getType() == ConnectivityManager.TYPE_MOBILE) {
                    isWifi = 1;
                } else if (info.getType() == ConnectivityManager.TYPE_WIFI) {
                    isWifi = 2;
                }
            }
            session.setNetworkType(isWifi);
        } catch (Exception e) {
            WiseLog.e("set network info error !!", e);
        }
    }

    private void setInch() {

        DisplayMetrics matrix = new DisplayMetrics();
        WindowManager wm = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        wm.getDefaultDisplay().getMetrics(matrix);

        int w = matrix.widthPixels;
        int h = matrix.heightPixels;

        float density = matrix.density;
        float xdpi = matrix.xdpi;
        float ydpi = matrix.ydpi;

        float x_inch = w / xdpi;
        float y_inch = h / ydpi;
        double display = Math.sqrt(x_inch * x_inch + y_inch * y_inch);
        session.setInch(Math.round(display * 10) / 10);

    }

    private void setUuid() {
        String uuid = session.getUuid();
        if (TextUtils.isEmpty(uuid)) {
            uuid = UUID.randomUUID().toString();
        }
        session.setUuid(uuid);
    }

    private void setDeviceUuid() {
        if (!checkSelfPermission(Manifest.permission.READ_PHONE_STATE)) {
            return;
        }
        try {
            TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(context.TELEPHONY_SERVICE);
            if (telephonyManager == null) {
                return;
            }
            String deviceUuid;
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                deviceUuid = telephonyManager.getImei();
            } else {
                deviceUuid = telephonyManager.getDeviceId();
            }
            if (TextUtils.isEmpty(deviceUuid)) {
                return;
            }
            session.setDeviceUuid(deviceUuid);
        } catch (Exception e) {
            WiseLog.e("set device uuid error", e);
        }
    }

    private void setVendorId() {
        if (!checkSelfPermission(Manifest.permission.READ_PHONE_STATE)) {
            return;
        }
        try {
            String vendorId;
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                vendorId = Build.getSerial();
            } else {
                vendorId = Build.SERIAL;
            }
            if (TextUtils.isEmpty(vendorId)) {
                return;
            }
            session.setVendorId(vendorId);
        } catch (Exception e) {
            WiseLog.e("set vendor id error", e);
        }
    }

    public void setSDKVersion() {
        Map<String, String> sdkVersionMap = new HashMap<>();
        sdkVersionMap.put("SESSION", BuildConfig.VERSION_NAME);
        session.setSdkVersion(sdkVersionMap);
    }

}
