/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.oss.driver.internal.core.pool;

import com.datastax.oss.driver.api.core.AsyncAutoCloseable;
import com.datastax.oss.driver.api.core.CqlIdentifier;
import com.datastax.oss.driver.api.core.InvalidKeyspaceException;
import com.datastax.oss.driver.api.core.UnsupportedProtocolVersionException;
import com.datastax.oss.driver.api.core.auth.AuthenticationException;
import com.datastax.oss.driver.api.core.config.DefaultDriverOption;
import com.datastax.oss.driver.api.core.config.DriverConfig;
import com.datastax.oss.driver.api.core.connection.ReconnectionPolicy;
import com.datastax.oss.driver.api.core.loadbalancing.NodeDistance;
import com.datastax.oss.driver.api.core.metadata.Node;
import com.datastax.oss.driver.api.core.metadata.token.Token;
import com.datastax.oss.driver.api.core.metrics.DefaultNodeMetric;
import com.datastax.oss.driver.internal.core.channel.ChannelEvent;
import com.datastax.oss.driver.internal.core.channel.ChannelFactory;
import com.datastax.oss.driver.internal.core.channel.ClusterNameMismatchException;
import com.datastax.oss.driver.internal.core.channel.DriverChannel;
import com.datastax.oss.driver.internal.core.channel.DriverChannelOptions;
import com.datastax.oss.driver.internal.core.config.ConfigChangeEvent;
import com.datastax.oss.driver.internal.core.context.EventBus;
import com.datastax.oss.driver.internal.core.context.InternalDriverContext;
import com.datastax.oss.driver.internal.core.metadata.DefaultNode;
import com.datastax.oss.driver.internal.core.metadata.TopologyEvent;
import com.datastax.oss.driver.internal.core.pool.ChannelSet;
import com.datastax.oss.driver.internal.core.protocol.ShardingInfo;
import com.datastax.oss.driver.internal.core.util.Loggers;
import com.datastax.oss.driver.internal.core.util.concurrent.CompletableFutures;
import com.datastax.oss.driver.internal.core.util.concurrent.Reconnection;
import com.datastax.oss.driver.internal.core.util.concurrent.RunOrSchedule;
import com.datastax.oss.driver.internal.core.util.concurrent.UncaughtExceptions;
import com.datastax.oss.driver.shaded.guava.common.annotations.VisibleForTesting;
import com.datastax.oss.driver.shaded.guava.common.collect.Sets;
import com.datastax.oss.driver.shaded.netty.util.concurrent.EventExecutor;
import com.datastax.oss.driver.shaded.netty.util.concurrent.Future;
import com.datastax.oss.driver.shaded.netty.util.concurrent.GenericFutureListener;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import net.jcip.annotations.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class ChannelPool
implements AsyncAutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(ChannelPool.class);
    @VisibleForTesting
    ChannelSet[] channels;
    private final Node node;
    private final CqlIdentifier initialKeyspaceName;
    private final EventExecutor adminExecutor;
    private final String sessionLogPrefix;
    private final String logPrefix;
    private final SingleThreaded singleThreaded;
    private volatile boolean invalidKeyspace;

    public static CompletionStage<ChannelPool> init(Node node, CqlIdentifier keyspaceName, NodeDistance distance, InternalDriverContext context, String sessionLogPrefix) {
        ChannelPool pool = new ChannelPool(node, keyspaceName, distance, context, sessionLogPrefix);
        return pool.connect();
    }

    private ChannelPool(Node node, CqlIdentifier keyspaceName, NodeDistance distance, InternalDriverContext context, String sessionLogPrefix) {
        this.node = node;
        this.initialKeyspaceName = keyspaceName;
        this.adminExecutor = context.getNettyOptions().adminEventExecutorGroup().next();
        this.sessionLogPrefix = sessionLogPrefix;
        this.logPrefix = sessionLogPrefix + "|" + node.getEndPoint();
        this.singleThreaded = new SingleThreaded(keyspaceName, distance, context);
    }

    private CompletionStage<ChannelPool> connect() {
        RunOrSchedule.on(this.adminExecutor, () -> this.singleThreaded.connect());
        return this.singleThreaded.connectFuture;
    }

    public Node getNode() {
        return this.node;
    }

    public CqlIdentifier getInitialKeyspaceName() {
        return this.initialKeyspaceName;
    }

    public boolean isInvalidKeyspace() {
        return this.invalidKeyspace;
    }

    public DriverChannel next() {
        return this.next(null, null);
    }

    public DriverChannel next(@Nullable Token routingKey, @Nullable Integer shardSuggestion) {
        int firstShardToCheck;
        if (!this.singleThreaded.initialized) {
            return null;
        }
        if (this.singleThreaded.shardingInfo == null) {
            return this.channels[0].next();
        }
        int shardId = -1;
        if (shardSuggestion != null) {
            if (shardSuggestion >= this.channels.length) {
                LOG.warn("Shard suggestion is out of channels array bounds. Ignoring.");
            } else {
                shardId = shardSuggestion;
            }
        }
        if (shardId == -1) {
            int n = shardId = routingKey != null ? this.singleThreaded.shardingInfo.shardId(routingKey) : ThreadLocalRandom.current().nextInt(this.channels.length);
        }
        if (this.channels[shardId].size() > 0) {
            return this.channels[shardId].next();
        }
        int shardToCheck = firstShardToCheck = ThreadLocalRandom.current().nextInt(this.channels.length);
        do {
            if (this.channels[shardToCheck].size() <= 0) continue;
            return this.channels[shardToCheck].next();
        } while ((shardToCheck = (shardToCheck + 1) % this.channels.length) != firstShardToCheck);
        return null;
    }

    public int size() {
        return Arrays.stream(this.channels).mapToInt(ChannelSet::size).sum();
    }

    public int getAvailableIds() {
        return Arrays.stream(this.channels).mapToInt(ChannelSet::getAvailableIds).sum();
    }

    public int getInFlight() {
        return Arrays.stream(this.channels).mapToInt(ChannelSet::getInFlight).sum();
    }

    public int getOrphanedIds() {
        return Arrays.stream(this.channels).mapToInt(ChannelSet::getOrphanedIds).sum();
    }

    public void resize(NodeDistance newDistance) {
        RunOrSchedule.on(this.adminExecutor, () -> this.singleThreaded.resize(newDistance));
    }

    public CompletionStage<Void> setKeyspace(CqlIdentifier newKeyspaceName) {
        return RunOrSchedule.on(this.adminExecutor, () -> this.singleThreaded.setKeyspace(newKeyspaceName));
    }

    public void reconnectNow() {
        RunOrSchedule.on(this.adminExecutor, () -> this.singleThreaded.reconnectNow());
    }

    @Override
    @NonNull
    public CompletionStage<Void> closeFuture() {
        return this.singleThreaded.closeFuture;
    }

    @Override
    @NonNull
    public CompletionStage<Void> closeAsync() {
        RunOrSchedule.on(this.adminExecutor, () -> this.singleThreaded.close());
        return this.singleThreaded.closeFuture;
    }

    @Override
    @NonNull
    public CompletionStage<Void> forceCloseAsync() {
        RunOrSchedule.on(this.adminExecutor, () -> this.singleThreaded.forceClose());
        return this.singleThreaded.closeFuture;
    }

    private class SingleThreaded {
        private final DriverConfig config;
        private final ChannelFactory channelFactory;
        private final EventBus eventBus;
        private final List<CompletionStage<DriverChannel>> pendingChannels = new ArrayList<CompletionStage<DriverChannel>>();
        private final Set<DriverChannel> closingChannels = new HashSet<DriverChannel>();
        private final Reconnection reconnection;
        private final Object configListenerKey;
        private NodeDistance distance;
        private volatile boolean initialized = false;
        private int wantedCount;
        private final CompletableFuture<ChannelPool> connectFuture = new CompletableFuture();
        private boolean isConnecting;
        private final CompletableFuture<Void> closeFuture = new CompletableFuture();
        private boolean isClosing;
        private CompletableFuture<Void> setKeyspaceFuture;
        private CqlIdentifier keyspaceName;
        private volatile ShardingInfo shardingInfo;

        private SingleThreaded(CqlIdentifier keyspaceName, NodeDistance distance, InternalDriverContext context) {
            this.keyspaceName = keyspaceName;
            this.config = context.getConfig();
            this.distance = distance;
            this.channelFactory = context.getChannelFactory();
            this.eventBus = context.getEventBus();
            ReconnectionPolicy reconnectionPolicy = context.getReconnectionPolicy();
            this.reconnection = new Reconnection(ChannelPool.this.logPrefix, ChannelPool.this.adminExecutor, () -> reconnectionPolicy.newNodeSchedule(ChannelPool.this.node), this::reconnect, () -> this.eventBus.fire(ChannelEvent.reconnectionStarted(ChannelPool.this.node)), () -> this.eventBus.fire(ChannelEvent.reconnectionStopped(ChannelPool.this.node)));
            this.configListenerKey = this.eventBus.register(ConfigChangeEvent.class, RunOrSchedule.on(ChannelPool.this.adminExecutor, this::onConfigChanged));
        }

        private DriverChannelOptions buildDriverOptions() {
            return DriverChannelOptions.builder().withKeyspace(this.keyspaceName).withOwnerLogPrefix(ChannelPool.this.sessionLogPrefix).build();
        }

        private void addChannel(DriverChannel c) {
            ChannelPool.this.channels[c.getShardId()].add(c);
            this.eventBus.fire(ChannelEvent.channelOpened(ChannelPool.this.node));
            c.closeStartedFuture().addListener((GenericFutureListener<? extends Future<? super Void>>)((GenericFutureListener<Future>)f -> ChannelPool.this.adminExecutor.submit(() -> this.onChannelCloseStarted(c)).addListener(UncaughtExceptions::log)));
            c.closeFuture().addListener((GenericFutureListener<? extends Future<? super Void>>)((GenericFutureListener<Future>)f -> ChannelPool.this.adminExecutor.submit(() -> this.onChannelClosed(c)).addListener(UncaughtExceptions::log)));
        }

        private void initialize(DriverChannel c) {
            this.shardingInfo = c.getShardingInfo();
            ((DefaultNode)ChannelPool.this.node).setShardingInfo(this.shardingInfo);
            int wanted = this.getConfiguredSize(this.distance);
            int shardsCount = this.shardingInfo == null ? 1 : this.shardingInfo.getShardsCount();
            this.wantedCount = wanted / shardsCount + (wanted % shardsCount > 0 ? 1 : 0);
            ChannelPool.this.channels = new ChannelSet[shardsCount];
            for (int i = 0; i < ChannelPool.this.channels.length; ++i) {
                ChannelPool.this.channels[i] = new ChannelSet();
            }
            this.addChannel(c);
            this.initialized = true;
        }

        private CompletionStage<Boolean> reconnect() {
            if (this.initialized) {
                return this.addMissingChannels();
            }
            CompletableFuture<Boolean> resultFuture = new CompletableFuture<Boolean>();
            DriverChannelOptions options = this.buildDriverOptions();
            CompletionStage<DriverChannel> channelFuture = this.channelFactory.connect(ChannelPool.this.node, options);
            this.pendingChannels.add(channelFuture);
            channelFuture.handleAsync((driver, e) -> {
                this.pendingChannels.clear();
                if (e != null) {
                    Throwable[] fatalError = new Throwable[1];
                    this.handleError((Throwable)e, t -> {
                        fatalError[0] = t;
                    }, ignored -> ChannelPool.this.invalidKeyspace = true);
                    if (fatalError[0] != null) {
                        Loggers.warnWithException(LOG, "[{}] Fatal error while initializing pool, forcing the node down", ChannelPool.this.logPrefix, fatalError[0]);
                        ChannelPool.this.node.getBroadcastRpcAddress().ifPresent(address -> this.eventBus.fire(TopologyEvent.forceDown(address)));
                        resultFuture.complete(true);
                    } else {
                        resultFuture.complete(false);
                    }
                } else {
                    this.initialize((DriverChannel)driver);
                    CompletableFutures.completeFrom(this.addMissingChannels(), resultFuture);
                }
                return null;
            }, ChannelPool.this.adminExecutor);
            return resultFuture;
        }

        private void makeInitialConnection() {
            DriverChannelOptions options = this.buildDriverOptions();
            CompletionStage<DriverChannel> channelFuture = this.channelFactory.connect(ChannelPool.this.node, options);
            this.pendingChannels.add(channelFuture);
            channelFuture.handleAsync((driver, e) -> {
                this.pendingChannels.clear();
                if (e != null) {
                    Throwable[] fatalError = new Throwable[1];
                    this.handleError((Throwable)e, t -> {
                        fatalError[0] = t;
                    }, ignored -> ChannelPool.this.invalidKeyspace = true);
                    if (fatalError[0] != null) {
                        Loggers.warnWithException(LOG, "[{}] Fatal error while initializing pool, forcing the node down", ChannelPool.this.logPrefix, fatalError[0]);
                        ChannelPool.this.node.getBroadcastRpcAddress().ifPresent(address -> this.eventBus.fire(TopologyEvent.forceDown(address)));
                    } else {
                        this.reconnection.start();
                    }
                    this.connectFuture.complete(ChannelPool.this);
                } else {
                    this.initialize((DriverChannel)driver);
                    CompletionStage<ChannelPool> initialChannels = this.addMissingChannels().thenApply(allConnected -> {
                        if (!allConnected.booleanValue()) {
                            this.reconnection.start();
                        }
                        return ChannelPool.this;
                    });
                    CompletableFutures.completeFrom(initialChannels, this.connectFuture);
                }
                return null;
            }, ChannelPool.this.adminExecutor);
        }

        private void connect() {
            assert (ChannelPool.this.adminExecutor.inEventLoop());
            if (this.isConnecting) {
                return;
            }
            this.isConnecting = true;
            this.makeInitialConnection();
        }

        private CompletionStage<Boolean> addMissingChannels() {
            assert (ChannelPool.this.adminExecutor.inEventLoop());
            assert (this.pendingChannels.isEmpty());
            int missing = ChannelPool.this.channels.length * this.wantedCount - Arrays.stream(ChannelPool.this.channels).mapToInt(ChannelSet::size).sum();
            LOG.debug("[{}] Trying to create {} missing channels", (Object)ChannelPool.this.logPrefix, (Object)missing);
            DriverChannelOptions options = this.buildDriverOptions();
            int batchSize = this.config.getDefaultProfile().getInt(DefaultDriverOption.CONNECTION_POOL_INIT_BATCH_SIZE);
            batchSize = Integer.min(batchSize, (ChannelPool.this.channels.length * this.wantedCount + 1) / 2);
            ArrayList<CompletionStage<Object>> previousBatch = new ArrayList();
            ArrayList currentBatch = new ArrayList();
            for (int shard = 0; shard < ChannelPool.this.channels.length; ++shard) {
                LOG.trace("[{}] Missing {} channels for shard {}", new Object[]{ChannelPool.this.logPrefix, this.wantedCount - ChannelPool.this.channels[shard].size(), shard});
                for (int p = ChannelPool.this.channels[shard].size(); p < this.wantedCount; ++p) {
                    CompletionStage channelFuture;
                    if (this.config.getDefaultProfile().getBoolean(DefaultDriverOption.CONNECTION_ADVANCED_SHARD_AWARENESS_ENABLED)) {
                        int finalShard = shard;
                        channelFuture = CompletableFutures.allDone(previousBatch).thenComposeAsync(ignored -> this.channelFactory.connect(ChannelPool.this.node, finalShard, options), ChannelPool.this.adminExecutor);
                    } else {
                        channelFuture = CompletableFutures.allDone(previousBatch).thenComposeAsync(ignored -> this.channelFactory.connect(ChannelPool.this.node, options), ChannelPool.this.adminExecutor);
                    }
                    this.pendingChannels.add(channelFuture);
                    if (batchSize == 0) continue;
                    currentBatch.add(channelFuture);
                    if (currentBatch.size() < batchSize) continue;
                    previousBatch = currentBatch;
                    currentBatch = new ArrayList();
                }
            }
            return CompletableFutures.allDone(this.pendingChannels).thenApplyAsync(this::onAllConnected, ChannelPool.this.adminExecutor);
        }

        private void handleError(Throwable error, Consumer<Throwable> onFatal, Consumer<Void> onKeyspaceError) {
            ((DefaultNode)ChannelPool.this.node).getMetricUpdater().incrementCounter(error instanceof AuthenticationException ? DefaultNodeMetric.AUTHENTICATION_ERRORS : DefaultNodeMetric.CONNECTION_INIT_ERRORS, null);
            if (error instanceof ClusterNameMismatchException || error instanceof UnsupportedProtocolVersionException) {
                onFatal.accept(error);
            } else if (error instanceof AuthenticationException) {
                Loggers.warnWithException(LOG, "[{}] Authentication error", ChannelPool.this.logPrefix, error);
            } else if (error instanceof InvalidKeyspaceException) {
                onKeyspaceError.accept(null);
            } else if (this.config.getDefaultProfile().getBoolean(DefaultDriverOption.CONNECTION_WARN_INIT_ERROR)) {
                Loggers.warnWithException(LOG, "[{}]  Error while opening new channel", ChannelPool.this.logPrefix, error);
            } else {
                LOG.debug("[{}]  Error while opening new channel", (Object)ChannelPool.this.logPrefix, (Object)error);
            }
        }

        private boolean onAllConnected(Void v) {
            assert (ChannelPool.this.adminExecutor.inEventLoop());
            Throwable[] fatalError = new Throwable[1];
            int[] invalidKeyspaceErrors = new int[]{0};
            for (CompletionStage<DriverChannel> pendingChannel : this.pendingChannels) {
                CompletableFuture<DriverChannel> future = pendingChannel.toCompletableFuture();
                assert (future.isDone());
                if (future.isCompletedExceptionally()) {
                    Throwable error = CompletableFutures.getFailed(future);
                    this.handleError(error, e -> {
                        fatalError[0] = e;
                    }, ignored -> {
                        invalidKeyspaceErrors[0] = invalidKeyspaceErrors[0] + 1;
                    });
                    continue;
                }
                DriverChannel channel = CompletableFutures.getCompleted(future);
                if (this.isClosing) {
                    LOG.debug("[{}] New channel added ({}) but the pool was closed, closing it", (Object)ChannelPool.this.logPrefix, (Object)channel);
                    channel.forceClose();
                    continue;
                }
                if (this.config.getDefaultProfile().getBoolean(DefaultDriverOption.CONNECTION_ADVANCED_SHARD_AWARENESS_ENABLED) && channel.localAddress() instanceof InetSocketAddress && channel.getShardingInfo() != null) {
                    int targetShard;
                    int port = ((InetSocketAddress)channel.localAddress()).getPort();
                    int actualShard = channel.getShardId();
                    if (actualShard != (targetShard = port % channel.getShardingInfo().getShardsCount())) {
                        LOG.warn("[{}] New channel {} connected to shard {}, but shard {} was requested. If this is not transient check your driver AND cluster configuration of shard aware port.", new Object[]{ChannelPool.this.logPrefix, channel, actualShard, targetShard});
                    }
                }
                LOG.debug("[{}] New channel added {}", (Object)ChannelPool.this.logPrefix, (Object)channel);
                if (ChannelPool.this.channels[channel.getShardId()].size() < this.wantedCount) {
                    this.addChannel(channel);
                    continue;
                }
                channel.close();
            }
            ChannelPool.this.invalidKeyspace = invalidKeyspaceErrors[0] > 0 && invalidKeyspaceErrors[0] == this.pendingChannels.size();
            this.pendingChannels.clear();
            if (fatalError[0] != null) {
                Loggers.warnWithException(LOG, "[{}] Fatal error while initializing pool, forcing the node down", ChannelPool.this.logPrefix, fatalError[0]);
                ChannelPool.this.node.getBroadcastRpcAddress().ifPresent(address -> this.eventBus.fire(TopologyEvent.forceDown(address)));
                return true;
            }
            this.shrinkIfTooManyChannels();
            int currentCount = Arrays.stream(ChannelPool.this.channels).mapToInt(ChannelSet::size).sum();
            LOG.debug("[{}] Reconnection attempt complete, {}/{} channels", new Object[]{ChannelPool.this.logPrefix, currentCount, ChannelPool.this.channels.length * this.wantedCount});
            return currentCount >= ChannelPool.this.channels.length * this.wantedCount;
        }

        private void onChannelCloseStarted(DriverChannel channel) {
            assert (ChannelPool.this.adminExecutor.inEventLoop());
            if (!this.isClosing) {
                LOG.debug("[{}] Channel {} started graceful shutdown", (Object)ChannelPool.this.logPrefix, (Object)channel);
                ChannelPool.this.channels[channel.getShardId()].remove(channel);
                this.closingChannels.add(channel);
                this.eventBus.fire(ChannelEvent.channelClosed(ChannelPool.this.node));
                this.reconnection.start();
            }
        }

        private void onChannelClosed(DriverChannel channel) {
            assert (ChannelPool.this.adminExecutor.inEventLoop());
            if (!this.isClosing) {
                if (ChannelPool.this.channels[channel.getShardId()].remove(channel)) {
                    LOG.debug("[{}] Lost channel {}", (Object)ChannelPool.this.logPrefix, (Object)channel);
                    this.eventBus.fire(ChannelEvent.channelClosed(ChannelPool.this.node));
                    this.reconnection.start();
                } else {
                    LOG.debug("[{}] Channel {} completed graceful shutdown", (Object)ChannelPool.this.logPrefix, (Object)channel);
                    this.closingChannels.remove(channel);
                }
            }
        }

        private void resize(NodeDistance newDistance) {
            assert (ChannelPool.this.adminExecutor.inEventLoop());
            this.distance = newDistance;
            int newChannelCount = this.getConfiguredSize(newDistance);
            int shardsCount = this.shardingInfo == null ? 1 : this.shardingInfo.getShardsCount();
            if ((newChannelCount = newChannelCount / shardsCount + (newChannelCount % shardsCount > 0 ? 1 : 0)) > this.wantedCount) {
                LOG.debug("[{}] Growing ({} => {} channels)", new Object[]{ChannelPool.this.logPrefix, this.wantedCount, newChannelCount});
                this.wantedCount = newChannelCount;
                this.reconnection.start();
            } else if (newChannelCount < this.wantedCount) {
                LOG.debug("[{}] Shrinking ({} => {} channels)", new Object[]{ChannelPool.this.logPrefix, this.wantedCount, newChannelCount});
                this.wantedCount = newChannelCount;
                if (!this.reconnection.isRunning()) {
                    this.shrinkIfTooManyChannels();
                }
            }
        }

        private void shrinkIfTooManyChannels() {
            assert (ChannelPool.this.adminExecutor.inEventLoop());
            if (this.initialized) {
                HashSet toRemove = Sets.newHashSet();
                for (int shardId = 0; shardId < ChannelPool.this.channels.length; ++shardId) {
                    int extra = ChannelPool.this.channels[shardId].size() - this.wantedCount;
                    if (extra <= 0) continue;
                    LOG.debug("[{}] Closing {} extra channels for shard {}", new Object[]{ChannelPool.this.logPrefix, extra, shardId});
                    for (DriverChannel channel : ChannelPool.this.channels[shardId]) {
                        toRemove.add(channel);
                        if (--extra != 0) continue;
                        break;
                    }
                    for (DriverChannel channel : toRemove) {
                        ChannelPool.this.channels[channel.getShardId()].remove(channel);
                        channel.close();
                        this.eventBus.fire(ChannelEvent.channelClosed(ChannelPool.this.node));
                    }
                    toRemove.clear();
                }
            }
        }

        private void onConfigChanged(ConfigChangeEvent event) {
            assert (ChannelPool.this.adminExecutor.inEventLoop());
            this.resize(this.distance);
        }

        private CompletionStage<Void> setKeyspace(CqlIdentifier newKeyspaceName) {
            int toSwitch;
            assert (ChannelPool.this.adminExecutor.inEventLoop());
            if (this.setKeyspaceFuture != null && !this.setKeyspaceFuture.isDone()) {
                return CompletableFutures.failedFuture(new IllegalStateException("Can't call setKeyspace while a keyspace switch is already in progress"));
            }
            this.keyspaceName = newKeyspaceName;
            this.setKeyspaceFuture = new CompletableFuture();
            int n = toSwitch = !this.initialized ? 0 : Arrays.stream(ChannelPool.this.channels).mapToInt(ChannelSet::size).sum();
            if (toSwitch == 0) {
                this.setKeyspaceFuture.complete(null);
            } else {
                AtomicInteger remaining = new AtomicInteger(toSwitch);
                for (ChannelSet set : ChannelPool.this.channels) {
                    for (DriverChannel channel2 : set) {
                        channel2.setKeyspace(newKeyspaceName).addListener(f -> {
                            if (remaining.decrementAndGet() == 0) {
                                this.setKeyspaceFuture.complete(null);
                            }
                        });
                    }
                }
            }
            for (CompletionStage<DriverChannel> channelFuture : this.pendingChannels) {
                channelFuture.thenAccept(channel -> channel.setKeyspace(newKeyspaceName));
            }
            return this.setKeyspaceFuture;
        }

        private void reconnectNow() {
            assert (ChannelPool.this.adminExecutor.inEventLoop());
            this.reconnection.reconnectNow(false);
        }

        private void close() {
            assert (ChannelPool.this.adminExecutor.inEventLoop());
            if (this.isClosing) {
                return;
            }
            this.isClosing = true;
            this.reconnection.stop();
            this.eventBus.unregister(this.configListenerKey, ConfigChangeEvent.class);
            int toClose = this.closingChannels.size() + (!this.initialized ? 0 : Arrays.stream(ChannelPool.this.channels).mapToInt(ChannelSet::size).sum());
            if (toClose == 0) {
                this.closeFuture.complete(null);
            } else {
                AtomicInteger remaining = new AtomicInteger(toClose);
                GenericFutureListener<Future> channelCloseListener = f -> {
                    if (!f.isSuccess()) {
                        Loggers.warnWithException(LOG, "[{}] Error closing channel", ChannelPool.this.logPrefix, f.cause());
                    }
                    if (remaining.decrementAndGet() == 0) {
                        this.closeFuture.complete(null);
                    }
                };
                if (this.initialized) {
                    for (ChannelSet set : ChannelPool.this.channels) {
                        Iterator<DriverChannel> iterator = set.iterator();
                        while (iterator.hasNext()) {
                            DriverChannel channel = iterator.next();
                            this.eventBus.fire(ChannelEvent.channelClosed(ChannelPool.this.node));
                            channel.close().addListener(channelCloseListener);
                        }
                    }
                }
                for (DriverChannel channel : this.closingChannels) {
                    channel.closeFuture().addListener((GenericFutureListener<? extends Future<? super Void>>)channelCloseListener);
                }
            }
        }

        private void forceClose() {
            assert (ChannelPool.this.adminExecutor.inEventLoop());
            if (!this.isClosing) {
                this.close();
            }
            if (this.initialized) {
                for (ChannelSet set : ChannelPool.this.channels) {
                    Iterator<DriverChannel> iterator = set.iterator();
                    while (iterator.hasNext()) {
                        DriverChannel channel = iterator.next();
                        channel.forceClose();
                    }
                }
            }
            for (DriverChannel channel : this.closingChannels) {
                channel.forceClose();
            }
        }

        private int getConfiguredSize(NodeDistance distance) {
            return this.config.getDefaultProfile().getInt(distance == NodeDistance.LOCAL ? DefaultDriverOption.CONNECTION_POOL_LOCAL_SIZE : DefaultDriverOption.CONNECTION_POOL_REMOTE_SIZE);
        }
    }
}

