package com.scansolutions.mrzscannerlib;

import android.content.Context;
import android.graphics.Point;
import android.hardware.Camera;
import android.util.DisplayMetrics;
import android.util.Size;
import android.view.Display;
import android.view.WindowManager;

import java.util.ArrayList;
import java.util.List;

import static android.content.Context.WINDOW_SERVICE;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_DOC_IMAGE_ID;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_DOC_IMAGE_ID_FRONT;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_DOC_IMAGE_PASSPORT;

class MRZUtils {

    static final int IGNORE_RESULT_CODE = -99;

    static Point getDisplaySize(Context context) {
        WindowManager wm = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        Display display = wm.getDefaultDisplay();
        final Point size = new Point();
        display.getSize(size);
        return size;
    }

    static int getRotation(Context context) {
        return ((WindowManager) context.getSystemService(WINDOW_SERVICE)).getDefaultDisplay().getRotation();
    }

    static MRZSize scaleToDisplayRatio(Point displaySize, MRZSize viewSize) {
        final float displayAR = displaySize.y / (float) displaySize.x;
        float contentWidth = viewSize.width;
        float contentHeight = viewSize.height;

        float widthScaled;
        float heightScaled;

        if (contentWidth * displayAR >= contentHeight) {
            heightScaled = contentWidth * displayAR;
            widthScaled = contentWidth;
        } else {
            widthScaled = contentHeight / displayAR;
            heightScaled = contentHeight;
        }

        return new MRZSize(widthScaled, heightScaled);
    }

    static int convertDpToPixel(float dp, Context context) {
        return Math.round(dp * ((float) context.getResources().getDisplayMetrics().densityDpi / DisplayMetrics.DENSITY_DEFAULT));
    }

    static MRZPercentageRect calculateScaledRect(float scanningRectX,
                                                 float scanningRectY,
                                                 float width,
                                                 float height,
                                                 float widthRatio,
                                                 float heightRatio,
                                                 boolean forceFullImageScan) {
        if (forceFullImageScan)
            return new MRZPercentageRect(0, 0, 100, 100);

        int rectX = (int) (((scanningRectX / 100.0f) * widthRatio + (1.0f - widthRatio) / 2.0f) * 100.0f);
        int rectY = (int) (((scanningRectY / 100.0f) * heightRatio + (1.0f - heightRatio) / 2.0f) * 100.0f);
        int rectWidth = (int) (width * widthRatio);
        int rectHeight = (int) (height * heightRatio);

        return new MRZPercentageRect(rectX, rectY, rectWidth, rectHeight);
    }

    static List<MRZSize> mapSizes(Size[] sizes) {
        List<MRZSize> resultSizes = new ArrayList<>(sizes.length);
        for (Size s : sizes) {
            resultSizes.add(new MRZSize(s.getWidth(), s.getHeight()));
        }

        return resultSizes;
    }

    static List<MRZSize> mapSizes(List<Camera.Size> sizes) {
        List<MRZSize> resultSizes = new ArrayList<>(sizes.size());
        for (Camera.Size s : sizes) {
            resultSizes.add(new MRZSize(s.width, s.height));
        }

        return resultSizes;
    }

    static Point getBestFitSize(List<MRZSize> sizes, float screenAR) {
        MRZSize bestFit = null;
        int difference = Integer.MAX_VALUE;

        int TARGET_WIDTH = (MRZCore.effortLevel == MRZEffortLevel.EFFORT_LEVEL_CASUAL) ? 1280 : 1920;

        final float ASPECT_TOLERANCE = 0.2f;

        CameraLogger.addLog("Preview sizes: ");

        for (MRZSize s : sizes) {
            CameraLogger.addLog(s.width + " x " + s.height);
        }

        for (MRZSize s : sizes) {
            float currentAR = Math.max(s.width, s.height) / Math.min(s.width, s.height);

            if (Math.abs(currentAR - screenAR) <= ASPECT_TOLERANCE) {
                int biggerSide = (int) Math.max(s.width, s.height);
                int currentDiff = Math.abs(biggerSide - TARGET_WIDTH);

                if (currentDiff < difference) {
                    difference = currentDiff;
                    bestFit = s;
                } else if (currentDiff == difference && bestFit != null) {
                    float bfAR = Math.max(bestFit.width, bestFit.height) / Math.min(bestFit.width, bestFit.height);
                    if (Math.abs(currentAR - screenAR) < Math.abs(bfAR - screenAR)) {
                        bestFit = s;
                    }
                }
            }
        }

        if (bestFit == null) {
            for (MRZSize s : sizes) {
                int biggerSide = (int) Math.max(s.width, s.height);
                int currentDiff = Math.abs(biggerSide - TARGET_WIDTH);

                if (currentDiff < difference) {
                    difference = currentDiff;
                    bestFit = s;
                } else if (currentDiff == difference && bestFit != null) {
                    float currentSmallAR = Math.max(s.width, s.height) / Math.min(s.width, s.height);
                    float bfSmallAR = Math.max(bestFit.width, bestFit.height) / Math.min(bestFit.width, bestFit.height);
                    if (Math.abs(currentSmallAR - screenAR) < Math.abs(bfSmallAR - screenAR)) {
                        bestFit = s;
                    }
                }
            }
        }

        CameraLogger.addLog("Best fit: " + bestFit.width + " x " + (int) bestFit.height);

        return new Point((int) bestFit.width, (int) bestFit.height);
    }

    static boolean isScannerTypeDoc(ScannerType scannerType) {
        return scannerType == SCANNER_TYPE_DOC_IMAGE_ID || scannerType == SCANNER_TYPE_DOC_IMAGE_PASSPORT || scannerType == SCANNER_TYPE_DOC_IMAGE_ID_FRONT;
    }

    static String errorCodeToString(int errorCode) {
        switch (errorCode) {
            case 0:
                return null;
            case -1:
                return "Error while parsing the licence.";
            case -2:
                return "Invalid licence. Please make sure the provided licence is copied correctly without any additional spaces or characters.";
            case -3:
                return "Invalid bundle ID. Please make sure that your bundle ID matches the bundle ID provided for creating this licence.";
            case -4:
                return "Invalid device model. This licence only works with a predefined device model.";
            case -5:
                return "Licence expired.";
            case -6:
                return "Invalid platform. This licence does not work on iOS.";

            default:
                return "Error while parsing the licence.";
        }
    }

    static boolean contains(final int[] array, final int v) {
        for (int i : array) {
            if (i == v) {
                return true;
            }
        }

        return false;
    }

}
