package com.scansolutions.mrzscannerlib;

import android.content.Context;
import android.graphics.Point;
import android.hardware.Camera;
import android.util.Log;
import android.view.Display;
import android.view.Surface;
import android.view.WindowManager;

import com.scansolutions.mrzscannerlib.open.CameraFacing;
import com.scansolutions.mrzscannerlib.open.OpenCamera;

import java.util.List;

/**
 * A class which deals with reading, parsing, and setting the camera parameters which are used to
 * configure the camera hardware.
 */
@SuppressWarnings("deprecation") // camera APIs
final class CameraConfigurationManager {

    private static final String TAG = "CameraConfiguration";

    private final Context context;
    private int cwRotationFromDisplayToCamera;
    private Point cameraResolution;
    private Point bestPreviewSize;

    CameraConfigurationManager(Context context) {
        this.context = context;
    }

    /**
     * Reads, one time, values from the camera that are needed by the app.
     */
    void initFromCameraParameters(OpenCamera camera) {
        Camera.Parameters parameters = camera.getCamera().getParameters();
        WindowManager manager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        Display display = manager.getDefaultDisplay();

        int displayRotation = display.getRotation();
        int cwRotationFromNaturalToDisplay;
        switch (displayRotation) {
            case Surface.ROTATION_0:
                cwRotationFromNaturalToDisplay = 0;
                break;
            case Surface.ROTATION_90:
                cwRotationFromNaturalToDisplay = 90;
                break;
            case Surface.ROTATION_180:
                cwRotationFromNaturalToDisplay = 180;
                break;
            case Surface.ROTATION_270:
                cwRotationFromNaturalToDisplay = 270;
                break;
            default:
                // Have seen this return incorrect values like -90
                if (displayRotation % 90 == 0) {
                    cwRotationFromNaturalToDisplay = (360 + displayRotation) % 360;
                } else {
                    CameraLogger.addLog("Bad rotation: " + displayRotation);

                    throw new IllegalArgumentException("Bad rotation: " + displayRotation);
                }
        }

        CameraLogger.addLog("Display at: " + cwRotationFromNaturalToDisplay);
        Log.i(TAG, "Display at: " + cwRotationFromNaturalToDisplay);

        int cwRotationFromNaturalToCamera = camera.getOrientation();
        CameraLogger.addLog("Camera at: " + cwRotationFromNaturalToCamera);
        Log.i(TAG, "Camera at: " + cwRotationFromNaturalToCamera);

        // Still not 100% sure about this. But acts like we need to flip this:
        if (camera.getFacing() == CameraFacing.FRONT) {
            cwRotationFromNaturalToCamera = (360 - cwRotationFromNaturalToCamera) % 360;
            CameraLogger.addLog("Front camera overriden to: " + cwRotationFromNaturalToCamera);
            Log.i(TAG, "Front camera overriden to: " + cwRotationFromNaturalToCamera);
        }

        cwRotationFromDisplayToCamera =
                (360 + cwRotationFromNaturalToCamera - cwRotationFromNaturalToDisplay) % 360;

        CameraLogger.addLog("Final display orientation: " + cwRotationFromDisplayToCamera);
        Log.i(TAG, "Final display orientation: " + cwRotationFromDisplayToCamera);
        int cwNeededRotation;
        if (camera.getFacing() == CameraFacing.FRONT) {
            CameraLogger.addLog("Compensating rotation for front camera");
            Log.i(TAG, "Compensating rotation for front camera");
            cwNeededRotation = (360 - cwRotationFromDisplayToCamera) % 360;
        } else {
            cwNeededRotation = cwRotationFromDisplayToCamera;
        }
        CameraLogger.addLog("Clockwise rotation from display to camera: " + cwNeededRotation);
        Log.i(TAG, "Clockwise rotation from display to camera: " + cwNeededRotation);

        Point theScreenResolution = new Point();
        display.getSize(theScreenResolution);
        CameraLogger.addLog("Screen resolution in current orientation: " + theScreenResolution);
        Log.i(TAG, "Screen resolution in current orientation: " + theScreenResolution);

        List<Camera.Size> previewSizes = parameters.getSupportedPreviewSizes();
        List<MRZSize> mappedSizes = MRZUtils.mapSizes(previewSizes);

        bestPreviewSize = cameraResolution = MRZUtils.getBestFitSize(mappedSizes, Math.max(display.getWidth(), display.getHeight()) / (float) Math.min(display.getWidth(), display.getHeight()));
        Log.i(TAG, "Camera resolution: " + cameraResolution);
        Log.i(TAG, "Best available preview size: " + bestPreviewSize);

        boolean isScreenPortrait = theScreenResolution.x < theScreenResolution.y;
        boolean isPreviewSizePortrait = bestPreviewSize.x < bestPreviewSize.y;

        Point previewSizeOnScreen;
        if (isScreenPortrait == isPreviewSizePortrait) {
            previewSizeOnScreen = bestPreviewSize;
        } else {
            //noinspection SuspiciousNameCombination
            previewSizeOnScreen = new Point(bestPreviewSize.y, bestPreviewSize.x);
        }

        CameraLogger.addLog("Preview size on screen: " + previewSizeOnScreen);
        Log.i(TAG, "Preview size on screen: " + previewSizeOnScreen);
    }

    void setDesiredCameraParameters(OpenCamera camera, boolean safeMode) {
        Camera theCamera = camera.getCamera();
        Camera.Parameters parameters = theCamera.getParameters();

        if (parameters == null) {
            CameraLogger.addLog("Device error: no camera parameters are available. Proceeding without configuration.");
            Log.w(TAG, "Device error: no camera parameters are available. Proceeding without configuration.");
            return;
        }

        CameraLogger.addLog("Initial camera parameters: " + parameters.flatten());
        Log.i(TAG, "Initial camera parameters: " + parameters.flatten());

        if (safeMode) {
            Log.w(TAG, "In camera config safe mode -- most settings will not be honored");
        }

        initializeTorch(parameters);

        parameters.setPreviewSize(bestPreviewSize.x, bestPreviewSize.y);

        theCamera.setParameters(parameters);

        theCamera.setDisplayOrientation(cwRotationFromDisplayToCamera);

        Camera.Parameters afterParameters = theCamera.getParameters();
        Camera.Size afterSize = afterParameters.getPreviewSize();
        if (afterSize != null && (bestPreviewSize.x != afterSize.width || bestPreviewSize.y != afterSize.height)) {
            CameraLogger.addLog("Camera said it supported " + bestPreviewSize.x + 'x' + bestPreviewSize.y +
                    ", but after setting it, preview size is " + afterSize.width + 'x' + afterSize.height);

            Log.w(TAG, "Camera said it supported preview size " + bestPreviewSize.x + 'x' + bestPreviewSize.y +
                    ", but after setting it, preview size is " + afterSize.width + 'x' + afterSize.height);
            bestPreviewSize.x = afterSize.width;
            bestPreviewSize.y = afterSize.height;
        }
    }

    Point getCameraResolution() {
        return cameraResolution;
    }

    boolean getTorchState(Camera camera) {
        if (camera != null) {
            Camera.Parameters parameters = camera.getParameters();
            if (parameters != null) {
                String flashMode = parameters.getFlashMode();
                return Camera.Parameters.FLASH_MODE_ON.equals(flashMode) ||
                        Camera.Parameters.FLASH_MODE_TORCH.equals(flashMode);
            }
        }
        return false;
    }

    void setTorch(Camera camera, boolean newSetting) {
        Camera.Parameters parameters = camera.getParameters();
        doSetTorch(parameters, newSetting);
        camera.setParameters(parameters);
    }

    private void initializeTorch(Camera.Parameters parameters) {
        doSetTorch(parameters, false);
    }

    private void doSetTorch(Camera.Parameters parameters, boolean newSetting) {
        CameraConfigurationUtils.setTorch(parameters, newSetting);
    }

}
