package com.scansolutions.mrzscannerlib;

import android.content.Context;
import android.hardware.Camera;
import android.os.Handler;
import android.os.Message;
import android.view.SurfaceView;
import android.widget.ImageButton;
import android.widget.ImageView;

import org.opencv.core.CvType;
import org.opencv.core.Mat;
import org.opencv.imgproc.Imgproc;

import java.io.IOException;

import static com.scansolutions.mrzscannerlib.MRZCore.STATE_SCANNING;
import static com.scansolutions.mrzscannerlib.MRZCore.STATE_STOPPED;

class Camera1Impl extends BaseCameraImpl {

    private final CameraManager cameraManager;
    private final Handler handler;
    private static final int MSG_DECODE = 16161;

    Camera1Impl(Context context,
                SurfaceView surfaceView,
                ImageButton btnFlash,
                MRZOverlay mrzOverlay,
                ImageView debugPreview,
                MRZScannerListener scannerListener,
                CameraInitListener cameraInitListener,
                ScannerType scannerType) {
        super(context, scannerListener, surfaceView, btnFlash, cameraInitListener, mrzOverlay, debugPreview, scannerType);

        cameraManager = new CameraManager(context);
        handler = new Handler(new Handler.Callback() {
            @Override
            public boolean handleMessage(Message msg) {
                if (msg.what == MSG_DECODE) {
                    decode((byte[]) msg.obj, msg.arg1, msg.arg2);
                }

                return false;
            }
        });
    }

    void initCamera() {
        CameraLogger.addLog("initCamera");

        try {
            cameraManager.openDriver(mSurfaceView.getHolder());
        } catch (IOException e) {
            CameraLogger.addLog("openDriver error: " + e.getMessage());
            e.printStackTrace();
            return;
        }

        Camera.CameraInfo cameraInfo = new Camera.CameraInfo();
        Camera.getCameraInfo(0, cameraInfo);
        mSensorOrientation = cameraInfo.orientation;

        int rotation = MRZUtils.getRotation(mContext);
        CameraLogger.addLog("rotation: " + rotation);

        cameraManager.updateOrientation(rotation);
        cameraManager.startPreview();

        if (MRZCore.scanningState == STATE_STOPPED) {
            MRZCore.scanningState = MRZCore.STATE_SCANNING;
            cameraManager.requestPreviewFrame(handler, MSG_DECODE);
        }

        if (!cameraManager.doesCameraMeetMinimum() && MRZCore.shouldWarnCamera) {
            cameraInitListener.warnIncompatibleCamera();
        }

        switchFlash(mrzCore.isTorchOn);
    }

    @Override
    void stopScanner() {
        MRZCore.scanningState = STATE_STOPPED;
        cameraManager.stopPreview();
        cameraManager.closeDriver();
    }

    @Override
    void pauseScanner() {
        MRZCore.scanningState = STATE_STOPPED;
        cameraManager.stopPreview();
    }

    @Override
    void resumeScanner() {
        cameraManager.startPreview();
        cameraManager.requestPreviewFrame(handler, MSG_DECODE);

        MRZCore.scanningState = STATE_SCANNING;
        switchFlash(mrzCore.isTorchOn);
    }

    @Override
    void switchFlash(Boolean on) {
        mrzCore.isTorchOn = on;
        cameraManager.setTorch(on);
        if (on) {
            btnFlash.setImageResource(R.drawable.amrz_ic_flash_on);
        } else {
            btnFlash.setImageResource(R.drawable.amrz_ic_ico_flash_off);
        }
    }

    private void decode(final byte[] data, final int width, final int height) {
        if (MRZCore.activeThreads >= MRZCore.maxThreads || MRZCore.scanningState == STATE_STOPPED) {
            return;
        }

        MRZCore.activeThreads++;

        new Thread(new Runnable() {

            @Override
            public void run() {
                Mat mYuv = new Mat(height + height / 2, width, CvType.CV_8UC1);
                mYuv.put(0, 0, data);

                Mat mRgba = new Mat();
                Imgproc.cvtColor(mYuv, mRgba, Imgproc.COLOR_YUV2BGRA_NV12, 3);
                mYuv.release();
                ImageUtils.rotateMat(mRgba, MRZUtils.getRotation(mContext), mSensorOrientation);

                mrzCore.scan(mRgba);
                mRgba.release();

                MRZCore.activeThreads--;
            }
        }).start();
    }

    @Override
    void release() {
    }

}
