package com.scansolutions.mrzscannerlib;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.graphics.ImageFormat;
import android.graphics.Matrix;
import android.graphics.Point;
import android.graphics.RectF;
import android.graphics.SurfaceTexture;
import android.hardware.camera2.CameraAccessException;
import android.hardware.camera2.CameraCaptureSession;
import android.hardware.camera2.CameraCharacteristics;
import android.hardware.camera2.CameraDevice;
import android.hardware.camera2.CameraManager;
import android.hardware.camera2.CaptureRequest;
import android.hardware.camera2.params.StreamConfigurationMap;
import android.media.Image;
import android.media.ImageReader;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.RequiresApi;
import android.util.Log;
import android.util.Size;
import android.view.Surface;
import android.view.TextureView;
import android.view.View;
import android.widget.ImageButton;
import android.widget.ImageView;

import org.opencv.android.OpenCVLoader;
import org.opencv.core.Mat;

import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;

import static android.content.Context.CAMERA_SERVICE;
import static android.media.session.PlaybackState.STATE_STOPPED;
import static com.scansolutions.mrzscannerlib.MRZCore.STATE_SCANNING;
import static com.scansolutions.mrzscannerlib.MRZCore.scanningState;

@RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
class Camera2Impl {

    private MRZOverlay mrzOverlay;

    interface CameraInitListener {
        void warnIncompatibleCamera();
    }

    private static final String TAG = Camera2Impl.class.getSimpleName();
    private CameraCaptureSession mCaptureSession;
    private CameraDevice mCameraDevice;
    private Size mPreviewSize;
    private ImageReader mImageReader;
    private CaptureRequest.Builder mPreviewRequestBuilder;
    private Semaphore mCameraOpenCloseLock = new Semaphore(1);

    private boolean mFlashSupported;

    private Context mContext;
    private String mCameraId;

    private MRZCore mrzCore;
    private int mSensorOrientation;

    private CameraInitListener cameraInitListener;

    private TextureView mTextureView;
    private ImageButton btnFlash;

    Camera2Impl(Context context,
                MRZScannerListener scannerListener,
                TextureView mTextureView,
                ImageButton btnFlash,
                CameraInitListener cameraInitListener,
                MRZOverlay mrzOverlay,
                ImageView debugPreview,
                ScannerType scannerType) {
        this.mrzOverlay = mrzOverlay;
        OpenCVLoader.initDebug();
        mContext = context;
        mrzCore = new MRZCore(mrzOverlay, debugPreview, scannerListener, context, scannerType);

        this.mTextureView = mTextureView;
        this.btnFlash = btnFlash;
        this.cameraInitListener = cameraInitListener;
    }

    final TextureView.SurfaceTextureListener mSurfaceTextureListener
            = new TextureView.SurfaceTextureListener() {

        @Override
        public void onSurfaceTextureAvailable(SurfaceTexture texture, int width, int height) {
            openCamera2(width, height);
        }

        @Override
        public void onSurfaceTextureSizeChanged(SurfaceTexture texture, int width, int height) {
            configureTransform(width, height);
        }

        @Override
        public boolean onSurfaceTextureDestroyed(SurfaceTexture texture) {
            return true;
        }

        @Override
        public void onSurfaceTextureUpdated(SurfaceTexture texture) {
        }

    };

    private final CameraDevice.StateCallback mStateCallback = new CameraDevice.StateCallback() {

        @Override
        public void onOpened(@NonNull CameraDevice cameraDevice) {
            // This method is called when the camera is opened.  We start camera preview here.
            mCameraOpenCloseLock.release();
            mCameraDevice = cameraDevice;
            createCamera2PreviewSession();
        }

        @Override
        public void onDisconnected(@NonNull CameraDevice cameraDevice) {
            mCameraOpenCloseLock.release();
            cameraDevice.close();
            mCameraDevice = null;
        }

        @Override
        public void onError(@NonNull CameraDevice cameraDevice, int error) {
            mCameraOpenCloseLock.release();
            cameraDevice.close();
            mCameraDevice = null;
//            MRZScanner.this.finish(); TODO: Handle this
        }

    };

    private final ImageReader.OnImageAvailableListener mOnImageAvailableListener = new ImageReader.OnImageAvailableListener() {

        @Override
        public void onImageAvailable(ImageReader reader) {
            final Image img = reader.acquireLatestImage();

            if (MRZCore.activeThreads >= MRZCore.maxThreads || MRZCore.scanningState == STATE_STOPPED /*|| isCameraAdjustingFocus */ || img == null) {
                if (img != null) {
                    img.close();
                }

                return;
            }

            MRZCore.activeThreads++;

            new Thread(new Runnable() {

                @Override
                public void run() {
                    Mat imageGrab = ImageUtils.imageToMat(img);
                    img.close();

                    imageGrab = ImageUtils.rotateMat(imageGrab, MRZUtils.getRotation(mContext), mSensorOrientation);

                    mrzCore.scan(imageGrab);

                    MRZCore.activeThreads--;
                }
            }).start();
        }
    };

    /**
     * Sets up member variables related to camera.
     *
     * @param width
     * @param height
     */
    private boolean setUpCameraOutputs(int width, int height) throws NullPointerException {
        boolean afSupported = false;
        boolean resSupported = false;

        CameraManager manager = (CameraManager) mContext.getSystemService(CAMERA_SERVICE);
        try {
            for (String cameraId : manager.getCameraIdList()) {
                CameraCharacteristics characteristics = manager.getCameraCharacteristics(cameraId);

                // We don't use the front facing camera
                Integer facing = characteristics.get(CameraCharacteristics.LENS_FACING);
                if (facing != null && facing == CameraCharacteristics.LENS_FACING_FRONT) {
                    continue;
                }

                StreamConfigurationMap map = characteristics.get(CameraCharacteristics.SCALER_STREAM_CONFIGURATION_MAP);
                if (map == null) {
                    continue;
                }

                Point displaySize = MRZUtils.getDisplaySize(mContext);
                int maxPreviewWidth = displaySize.x;
                int maxPreviewHeight = displaySize.y;

                Size[] sizes = map.getOutputSizes(ImageFormat.YUV_420_888);
                List<MRZSize> mappedSizes = MRZUtils.mapSizes(sizes);
                Point bestFit = MRZUtils.getBestFitSize(mappedSizes, Math.max(maxPreviewWidth, maxPreviewHeight) / (float) Math.min(maxPreviewWidth, maxPreviewHeight));

                for (Size size : sizes)
                    if (size.getWidth() * size.getHeight() > 4500000) {
                        resSupported = true;
                        break;
                    }

                int[] afModes = characteristics.get(CameraCharacteristics.CONTROL_AF_AVAILABLE_MODES);
                if (afModes != null && afModes.length > 1) afSupported = true;

                mImageReader = ImageReader.newInstance(bestFit.x, bestFit.y, ImageFormat.YUV_420_888, MRZCore.maxThreads + 2);
                mImageReader.setOnImageAvailableListener(mOnImageAvailableListener, null);

                //noinspection ConstantConditions
                mSensorOrientation = characteristics.get(CameraCharacteristics.SENSOR_ORIENTATION);

                // Danger, W.R.! Attempting to use too large a preview size could  exceed the camera
                // bus' bandwidth limitation, resulting in gorgeous previews but the storage of
                // garbage capture data.
                mPreviewSize = new Size(bestFit.x, bestFit.y);
                /*chooseOptimalSize(map.getOutputSizes(SurfaceTexture.class), rotatedPreviewWidth,
                        rotatedPreviewHeight, maxPreviewWidth, maxPreviewHeight, new Size(bestFit.x, bestFit.y));*/

                // Check if the flash is supported.
                Boolean available = characteristics.get(CameraCharacteristics.FLASH_INFO_AVAILABLE);
                mFlashSupported = available == null ? false : available;
                mCameraId = cameraId;

                if (mFlashSupported) {
                    btnFlash.setVisibility(View.VISIBLE);
                    btnFlash.setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            switchFlash(!mrzCore.isTorchOn);
                        }
                    });
                }
            }
        } catch (CameraAccessException e) {
            e.printStackTrace();
        }

        return afSupported && resSupported;
    }

    private boolean cameraOpened = false;

    @SuppressLint("MissingPermission")
    void openCamera2(int width, int height) {
        if (!cameraOpened) {
            cameraOpened = true;

            boolean doesCameraMeetRequirements = setUpCameraOutputs(width, height);
            configureTransform(width, height);

            CameraManager manager = (CameraManager) mContext.getSystemService(CAMERA_SERVICE);
            try {
                if (!mCameraOpenCloseLock.tryAcquire(2500, TimeUnit.MILLISECONDS)) return;

                manager.openCamera(mCameraId, mStateCallback, null);
            } catch (CameraAccessException e) {
                e.printStackTrace();
            } catch (InterruptedException e) {
                throw new RuntimeException("Interrupted while trying to lock camera opening.", e);
            }

            if (!doesCameraMeetRequirements && MRZCore.shouldWarnCamera) {
                cameraInitListener.warnIncompatibleCamera();
            }
        }
    }

    void pauseScanning() {
        try {
            if (null != mCaptureSession) {
                mCaptureSession.stopRepeating();
            }
        } catch (CameraAccessException | IllegalStateException e) {
            e.printStackTrace();
        }
    }

    void resumeScanning() {
        if (mCaptureSession != null) {
            try {
                if (mrzCore.isTorchOn) {
                    mPreviewRequestBuilder.set(CaptureRequest.FLASH_MODE, CaptureRequest.FLASH_MODE_TORCH);
                }

                mCaptureSession.setRepeatingRequest(mPreviewRequestBuilder.build(), null, null);
            } catch (CameraAccessException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Closes the current {@link CameraDevice}.
     */
    void closeCamera2() {
        cameraOpened = false;
        scanningState = STATE_STOPPED;
        try {
            mCameraOpenCloseLock.acquire();
            if (null != mCaptureSession) {
                mCaptureSession.close();
                mCaptureSession = null;
            }
            if (null != mCameraDevice) {
                mCameraDevice.close();
                mCameraDevice = null;
            }
            if (null != mImageReader) {
                mImageReader.close();
                mImageReader = null;
            }
        } catch (InterruptedException e) {
            throw new RuntimeException("Interrupted while trying to lock camera closing.", e);
        } finally {
            mCameraOpenCloseLock.release();
        }
    }

    /**
     * Creates a new {@link CameraCaptureSession} for camera preview.
     */
    private void createCamera2PreviewSession() {
        try {
            SurfaceTexture texture = mTextureView.getSurfaceTexture();
            if (texture == null || mImageReader == null)
                return;

            // We configure the size of default buffer to be the size of camera preview we want.
            texture.setDefaultBufferSize(mPreviewSize.getWidth(), mPreviewSize.getHeight());

            // This is the output Surface we need to start preview.
            Surface surface = new Surface(texture);

            // We set up a CaptureRequest.Builder with the output Surface.
            mPreviewRequestBuilder = mCameraDevice.createCaptureRequest(CameraDevice.TEMPLATE_PREVIEW);
            mPreviewRequestBuilder.addTarget(surface);
            mPreviewRequestBuilder.addTarget(mImageReader.getSurface());

            // Here, we create a CameraCaptureSession for camera preview.
            mCameraDevice.createCaptureSession(Arrays.asList(surface, mImageReader.getSurface()), new CameraCaptureSession.StateCallback() {

                @Override
                public void onConfigured(@NonNull CameraCaptureSession cameraCaptureSession) {
                    // The camera is already closed
                    if (null == mCameraDevice) {
                        return;
                    }

                    // When the session is ready, we start displaying the preview.
                    mCaptureSession = cameraCaptureSession;
                    try {
                        // Auto focus should be continuous for camera preview.
                        mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AF_MODE, CaptureRequest.CONTROL_AF_MODE_CONTINUOUS_PICTURE);
                        if (mFlashSupported && mrzCore.isTorchOn) {
                            mPreviewRequestBuilder.set(CaptureRequest.FLASH_MODE, CaptureRequest.FLASH_MODE_TORCH);
                            btnFlash.setImageResource(R.drawable.amrz_ic_flash_on);
                        }

                        mCaptureSession.setRepeatingRequest(mPreviewRequestBuilder.build(), null, null);
                        MRZCore.scanningState = STATE_SCANNING;
                    } catch (CameraAccessException e) {
                        e.printStackTrace();
                    } catch (IllegalStateException e) {
                        e.printStackTrace();
                    }
                }

                @Override
                public void onConfigureFailed(@NonNull CameraCaptureSession cameraCaptureSession) {
                    Log.i(TAG, "onConfigureFailed");
                }
            }, null);
        } catch (CameraAccessException e) {
            e.printStackTrace();
        }
    }

    /**
     * Configures the necessary {@link android.graphics.Matrix} transformation to `mTextureView`.
     * This method should be called after the camera preview size is determined in
     * setUpCameraOutputs and also the size of `mTextureView` is fixed.
     *
     * @param viewWidth  The width of `mTextureView`
     * @param viewHeight The height of `mTextureView`
     */
    private void configureTransform(int viewWidth, int viewHeight) {
        if (null == mTextureView || null == mPreviewSize) {
            return;
        }
        int rotation = ((Activity) mContext).getWindowManager().getDefaultDisplay().getRotation();
        Matrix matrix = new Matrix();
        RectF viewRect = new RectF(0, 0, viewWidth, viewHeight);
        RectF bufferRect = new RectF(0, 0, mPreviewSize.getHeight(), mPreviewSize.getWidth());
        float centerX = viewRect.centerX();
        float centerY = viewRect.centerY();
        if (Surface.ROTATION_90 == rotation || Surface.ROTATION_270 == rotation) {
            bufferRect.offset(centerX - bufferRect.centerX(), centerY - bufferRect.centerY());
            matrix.setRectToRect(viewRect, bufferRect, Matrix.ScaleToFit.FILL);
            float scale = Math.max(
                    (float) viewHeight / mPreviewSize.getHeight(),
                    (float) viewWidth / mPreviewSize.getWidth());
            matrix.postScale(scale, scale, centerX, centerY);
            matrix.postRotate(90 * (rotation - 2), centerX, centerY);
        } else if (Surface.ROTATION_180 == rotation) {
            matrix.postRotate(180, centerX, centerY);
        }
        mTextureView.setTransform(matrix);
    }

    void switchFlash(Boolean flashState) {
        mrzCore.isTorchOn = flashState;

        if (mFlashSupported && mCaptureSession != null) {
            try {
                if (flashState) {
                    mPreviewRequestBuilder.set(CaptureRequest.FLASH_MODE, CaptureRequest.FLASH_MODE_TORCH);
                    btnFlash.setImageResource(R.drawable.amrz_ic_flash_on);
                } else {
                    mPreviewRequestBuilder.set(CaptureRequest.FLASH_MODE, CaptureRequest.FLASH_MODE_OFF);
                    btnFlash.setImageResource(R.drawable.amrz_ic_ico_flash_off);
                }

                mCaptureSession.setRepeatingRequest(mPreviewRequestBuilder.build(), null, null);
            } catch (CameraAccessException e) {
                e.printStackTrace();
            } catch (IllegalStateException e) {
                e.printStackTrace();
            }
        }
    }

    void setScannerType(ScannerType scannerType) {
        if (mrzCore != null) {
            mrzCore.setScannerType(scannerType);
        }
    }

}
