package com.scansolutions.mrzscannerlib;

import android.content.Context;
import android.graphics.Bitmap;
import android.hardware.camera2.CameraCharacteristics;
import android.hardware.camera2.CameraMetadata;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.widget.ImageView;

import org.opencv.android.Utils;
import org.opencv.core.CvException;
import org.opencv.core.CvType;
import org.opencv.core.Mat;

import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_DOC_IMAGE_PASSPORT;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_MRZ;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_SWISS_DRIVING_LICENCE;


class MRZCore {

    private static final String TAG = MRZCore.class.getSimpleName();

    static {
        System.loadLibrary("mrz-native-lib");
    }

    boolean isTorchOn = false;
    private ScannerType scannerType = SCANNER_TYPE_MRZ;

    MRZCore(MRZOverlay mrzOverlay, ImageView debugPreview, MRZScannerListener scannerListener, Context context, ScannerType scannerType) {
        this.mrzOverlay = mrzOverlay;
        this.debugPreview = debugPreview;
        this.scannerListener = scannerListener;
        this.scannerType = scannerType;
        TesseractHelper.prepareTesseract(context);
    }

    static native String mrzString(long addrInputImage,
                                   int[] squareArr,
                                   boolean forceFixImage,
                                   String path,
                                   boolean isIDActive,
                                   boolean isPassportActive,
                                   boolean isVisaActive,
                                   float scanningRectX,
                                   float scanningRectY,
                                   float scanningRectWidth,
                                   float scanningRectHeight,
//                                   long addrDebugImage,
                                   int scannerMode,
                                   boolean enableUpsideDown);

    static native String sdkVersionNative();

    static native int registerWithLicenseKey(Context context, String key, String bundleID, String modelName, String manufacturer);

    static native long shouldCallBack(String key);

    static native String getLid(String key);

    static native String encryptCommunication(String rawString);

    static native String decryptCommunication(String encryptedString);

    static native String getAndroidID(Context context);

    static native void setValidCountryCodes(String[] strings);

    static native void setValidateCountryCodesEnabled(boolean enabled);

    static native int faceDetection(long addrInputImage,
                                    long addrFaceOutputImage,
                                    long addrSignatureOutputImage,
                                    long addrPassportOutputImage,
                                    String directory,
                                    String countryCode,
                                    int[] squarePoints);

    static native int[] findDocumentImageNative(long addrInputImage, boolean isPassport);

    private MRZScannerListener scannerListener;

    static boolean isIDActive = true;
    static boolean isPassportActive = true;
    static boolean isVisaActive = true;

    static boolean enableUpsideDown = false;

    static boolean shouldWarnCamera = true;

    static final int STATE_SCANNING = 0;
    static final int STATE_STOPPED = 1;
    static final int STATE_FINDING_SQUARE = 2;

    static float scanningRectX = 1;
    static float scanningRectY = 30;
    static float width = 98;
    static float height = 40;
    static float widthRatio = 1;
    static float heightRatio = 1;

    static int scanningState = STATE_STOPPED;

    static int activeThreads = 0;

    static final int AVAILABLE_THREADS = Runtime.getRuntime().availableProcessors();
    static int maxThreads = Math.min(2, AVAILABLE_THREADS);
    private MRZOverlay mrzOverlay;
    private ImageView debugPreview;

    private static long lastRes;

    void scan(Mat imageGrab) {
        if (scannerType == SCANNER_TYPE_MRZ) {
            scanForMRZ(imageGrab, true, 0);
        } else if (scannerType == SCANNER_TYPE_SWISS_DRIVING_LICENCE) {
            scanForMRZ(imageGrab, true, 2);
        } else if (MRZUtils.isScannerTypeDoc(scannerType)) {
            scanForDocumentImage(imageGrab);
        }
    }

    private void scanForDocumentImage(Mat imageGrab) {
        int imageWidth = imageGrab.cols();
        int imageHeight = imageGrab.rows();

        if (scanningState == STATE_STOPPED)
            return;

        int[] locationSquare = findDocumentImageNative(imageGrab.getNativeObjAddr(), scannerType == SCANNER_TYPE_DOC_IMAGE_PASSPORT);

        if (mrzOverlay != null)
            mrzOverlay.drawLocationOverlay(locationSquare, imageWidth, imageHeight);

        if (locationSquare.length == 8 && (scanningState == STATE_FINDING_SQUARE || scannerType == SCANNER_TYPE_DOC_IMAGE_PASSPORT)) {
            if (scanningState != STATE_STOPPED) {
                scanningState = STATE_STOPPED;

                final Bitmap bitmapImg = ImageUtils.getBitmap(imageGrab);

                new Handler(Looper.getMainLooper()).post(new Runnable() {
                    @Override
                    public void run() {
                        if (scannerListener != null) {
                            scannerListener.successfulScanWithDocumentImage(bitmapImg);
                        }
                    }
                });
            }
        }
    }

    void scanForMRZ(Bitmap bmp) {
        Mat m = new Mat(bmp.getWidth(), bmp.getHeight(), CvType.CV_8U);
        Utils.bitmapToMat(bmp, m);

        boolean success = false;
        scanningState = STATE_SCANNING;

        for (int i = 0; i < 10; i++) {
            success = scanForMRZ(m, i % 2 != 0, 1, true);
            if (success)
                break;
        }

        if (!success) {
            MRZCore.triggerScanImageFailedOnMainThread(scannerListener);
        }
    }

    private boolean scanForMRZ(Mat imageGrab, boolean forceFixImage, int scannerMode) {
        return scanForMRZ(imageGrab, forceFixImage, scannerMode, false);
    }

    private boolean scanForMRZ(Mat imageGrab, boolean forceFixImage, int scannerMode, boolean forceFullImageScan) {
        MRZPercentageRect scaledScanningRect = MRZUtils.calculateScaledRect(scanningRectX, scanningRectY, width, height, widthRatio, heightRatio, forceFullImageScan);

        int[] intArr = new int[8];
        String result = mrzString(
                imageGrab.getNativeObjAddr(),
                intArr,
                forceFixImage,
                TesseractHelper.getDataPath(),
                isIDActive,
                isPassportActive,
                isVisaActive,
                scaledScanningRect.x,
                scaledScanningRect.y,
                scaledScanningRect.width,
                scaledScanningRect.height,
//                debugMat.getNativeObjAddr(),
                scannerMode,
                enableUpsideDown);

        if (mrzOverlay != null)
            mrzOverlay.drawLocationOverlay(intArr, imageGrab.cols(), imageGrab.rows());

//        if (BuildConfig.DEBUG) {
//            if (debugMat.cols() > 0 && debugMat.rows() > 0) {
//                final Bitmap bitmapImg = Bitmap.createBitmap(debugMat.cols(), debugMat.rows(), Bitmap.Config.ARGB_8888);
//                Utils.matToBitmap(debugMat, bitmapImg);
//                new Handler(Looper.getMainLooper()).post(new Runnable() {
//                    @Override
//                    public void run() {
//                        debugPreview.setImageBitmap(bitmapImg);
//                    }
//                });
//            }
//        }

        if (!result.isEmpty()) {
            try {
                if (scanningState != STATE_STOPPED) {
                    long currentRes = System.currentTimeMillis();

                    if (currentRes - lastRes < 150)
                        return false;

                    lastRes = currentRes;
                    scanningState = STATE_STOPPED;

                    long dateScanned = System.currentTimeMillis() / 1000;
                    final MRZResultModel resultModel = new MRZResultModel(result, dateScanned);

                    if (resultModel.documentTypeRaw.startsWith("P")) {
                        PassportImages images = scanForFaceAndSignature(imageGrab, resultModel.issuingCountry, intArr);
                        resultModel.portrait = images.face;
                        resultModel.signature = images.signature;
                        resultModel.fullImage = images.passport;
                    }

                    new Handler(Looper.getMainLooper()).post(new Runnable() {
                        @Override
                        public void run() {
                            if (scannerListener != null) {
                                scannerListener.successfulScanWithResult(resultModel);
                            }
                        }
                    });
                    return true;
                }
            } catch (CvException e) {
                Log.d("Exception", "Error message: " + e.getMessage());
            }
        }

        return false;
    }

    private PassportImages scanForFaceAndSignature(Mat inputImage, String issuingCountry, int[] squreArr) {
        PassportImages result = new PassportImages();
        Mat faceMat = new Mat();
        Mat signatureMat = new Mat();
        Mat passportMat = new Mat();

        faceDetection(inputImage.getNativeObjAddr(),
                faceMat.getNativeObjAddr(),
                signatureMat.getNativeObjAddr(),
                passportMat.getNativeObjAddr(),
                TesseractHelper.getDataPath(),
                issuingCountry,
                squreArr);

        result.face = ImageUtils.getBitmap(faceMat);
        result.signature = ImageUtils.getBitmap(signatureMat);
        result.passport = ImageUtils.getBitmap(passportMat);

        return result;
    }

    static void triggerScanImageFailedOnMainThread(final MRZScannerListener scannerListener) {
        new Handler(Looper.getMainLooper()).post(new Runnable() {
            @Override
            public void run() {
                scannerListener.scanImageFailed();
            }
        });
    }

    static boolean allowCamera2Support(Context context) {
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP) {
            android.hardware.camera2.CameraManager manager = (android.hardware.camera2.CameraManager) context.getSystemService(Context.CAMERA_SERVICE);
            if (manager != null) {
                try {

                    String cameraIdS = manager.getCameraIdList()[0];
                    CameraCharacteristics characteristics = manager.getCameraCharacteristics(cameraIdS);
                    int support = characteristics.get(CameraCharacteristics.INFO_SUPPORTED_HARDWARE_LEVEL);
                    return support == CameraMetadata.INFO_SUPPORTED_HARDWARE_LEVEL_LIMITED || support == CameraMetadata.INFO_SUPPORTED_HARDWARE_LEVEL_FULL;
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        return false;
    }

    public void setScannerType(ScannerType scannerType) {
        this.scannerType = scannerType;
    }

}
