package com.scansolutions.mrzscannerlib;

import android.app.Activity;
import android.content.Context;
import android.hardware.Camera;
import android.os.Handler;
import android.os.Message;
import android.view.Display;
import android.view.SurfaceHolder;
import android.view.SurfaceView;
import android.view.View;
import android.view.WindowManager;
import android.widget.ImageButton;
import android.widget.ImageView;

import org.opencv.android.OpenCVLoader;
import org.opencv.core.CvType;
import org.opencv.core.Mat;
import org.opencv.imgproc.Imgproc;

import java.io.IOException;

import static android.content.Context.WINDOW_SERVICE;
import static com.scansolutions.mrzscannerlib.MRZCore.STATE_SCANNING;
import static com.scansolutions.mrzscannerlib.MRZCore.STATE_STOPPED;

class Camera1Impl implements SurfaceHolder.Callback {
    private boolean hasSurface;
    private CameraManager cameraManager;
    private Handler handler;
    private SurfaceHolder surfaceHolder;
    private MRZCore mrzCore;
    private Context mContext;
    private int mSensorRotation;
    private ImageButton btnFlash;
    private static final int MSG_DECODE = 16161;

    private Camera2Impl.CameraInitListener cameraInitListener;

    Camera1Impl(Context context,
                SurfaceView surfaceView,
                ImageButton btnFlash,
                MRZOverlay mrzOverlay,
                ImageView debugPreview,
                MRZScannerListener scannerListener,
                Camera2Impl.CameraInitListener cameraInitListener) {
        OpenCVLoader.initDebug();
        this.cameraInitListener = cameraInitListener;
        mContext = context;
        mrzCore = new MRZCore(mrzOverlay, debugPreview, scannerListener, context);
        surfaceHolder = surfaceView.getHolder();
        hasSurface = false;
        cameraManager = new CameraManager(context);

        handler = new Handler(new Handler.Callback() {
            @Override
            public boolean handleMessage(Message msg) {
                if (msg.what == MSG_DECODE) {
                    decode((byte[]) msg.obj, msg.arg1, msg.arg2);
                }

                return false;
            }
        });
        this.btnFlash = btnFlash;

        this.btnFlash.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                toggleFlash(!mrzCore.isTorchOn);
            }
        });
    }

    void resume() {
        if (hasSurface) {
            initCamera();
        } else {
            // Install the callback and wait for surfaceCreated() to resume the
            // camera.
            surfaceHolder.addCallback(this);
        }

        if (!mrzCore.isTorchOn) {
            cameraManager.setTorch(false);
        }
    }

    @Override
    public void surfaceCreated(SurfaceHolder holder) {
        if (!hasSurface) {
            hasSurface = true;
        }
    }

    @Override
    public void surfaceChanged(SurfaceHolder holder, int format, int width, int height) {
        if (!((Activity) mContext).isFinishing())
            initCamera();
    }

    @Override
    public void surfaceDestroyed(SurfaceHolder holder) {
        hasSurface = false;
    }

    void initCamera() {
        try {
            cameraManager.openDriver(surfaceHolder);
        } catch (IOException e) {
            e.printStackTrace();
        }

        Camera.CameraInfo cameraInfo = new Camera.CameraInfo();
        Camera.getCameraInfo(0, cameraInfo);
        mSensorRotation = cameraInfo.orientation;

        Display display = ((WindowManager) mContext.getSystemService(WINDOW_SERVICE)).getDefaultDisplay();
        int rotation = display.getRotation();

        cameraManager.updateOrientation(rotation);
        cameraManager.startPreview();

        if (MRZCore.scanningState == STATE_STOPPED) {
            MRZCore.scanningState = MRZCore.STATE_SCANNING;
            cameraManager.requestPreviewFrame(handler, MSG_DECODE);
        }

        if (!cameraManager.doesCameraMeetMinimum() && MRZCore.shouldWarnCamera) {
            cameraInitListener.warnIncompatibleCamera();
        }

        toggleFlash(mrzCore.isTorchOn);
    }

    void stopScanner() {
        MRZCore.scanningState = STATE_STOPPED;
        cameraManager.stopPreview();
        cameraManager.closeDriver();
    }

    void pauseScanner() {
        MRZCore.scanningState = STATE_STOPPED;
        cameraManager.stopPreview();
    }

    void resumeScanner() {
        cameraManager.startPreview();
        cameraManager.requestPreviewFrame(handler, MSG_DECODE);

        MRZCore.scanningState = STATE_SCANNING;
        toggleFlash(mrzCore.isTorchOn);
    }

    void toggleFlash(Boolean on) {
        mrzCore.isTorchOn = on;
        cameraManager.setTorch(on);
        if (on) {
            btnFlash.setImageResource(R.drawable.amrz_ic_flash_on);
        } else {
            btnFlash.setImageResource(R.drawable.amrz_ic_ico_flash_off);
        }
    }

    private void decode(final byte[] data, final int width, final int height) {
        if (MRZCore.activeThreads >= MRZCore.maxThreads || MRZCore.scanningState == STATE_STOPPED) {
            return;
        }

        MRZCore.activeThreads++;

        new Thread(new Runnable() {

            @Override
            public void run() {
                Mat mYuv = new Mat(height + height / 2, width, CvType.CV_8UC1);
                mYuv.put(0, 0, data);

                Mat mRgba = new Mat();
                Imgproc.cvtColor(mYuv, mRgba, Imgproc.COLOR_YUV2BGRA_NV12, 3);
                mRgba = mrzCore.rotateMat(mRgba, mContext, mSensorRotation);

                mrzCore.scan(mRgba);

                MRZCore.activeThreads--;
            }
        }).start();
    }
}
