package com.scansolutions.mrzscannerlib;

import android.Manifest;
import android.annotation.TargetApi;
import android.app.Activity;
import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.v4.app.ActivityCompat;

import static android.os.Build.VERSION_CODES.M;

@TargetApi(M)
public class PermissionsHelper extends Activity implements ActivityCompat.OnRequestPermissionsResultCallback {

    private static final String ARG_PERMISSIONS = "ARG_PERMISSIONS";

    interface PermissionsHelperCallback {
        void onRequestedPermissionsGranted(boolean granted);
    }

    static final int PERMISSION_CAMERA = 1111;
    static final int PERMISSION_GALLERY = 1222;

    private static PermissionsHelperCallback callback;

    static void start(final Context context, int permissions_mode, PermissionsHelperCallback callback) {
        PermissionsHelper.callback = callback;
        final Intent intent = new Intent(context, PermissionsHelper.class)
                .putExtra(ARG_PERMISSIONS, permissions_mode)
                .addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        context.startActivity(intent);
    }

    private void handleIntent(final Intent intent) {
        final int requestCode = intent.getIntExtra(ARG_PERMISSIONS, 0);
        requestPermission(requestCode);
    }

    public static boolean hasPermissions(Context c, String... permissions) {
        if (permissions == null) return false;

        for (String permission : permissions) {
            if (ActivityCompat.checkSelfPermission(c, permission) != PackageManager.PERMISSION_GRANTED) {
                return false;
            }
        }

        return true;
    }

    void requestPermission(final int requestCode) {
        if (Build.VERSION.SDK_INT >= M) {
            boolean shouldShowRationaleGallery = requestCode == PERMISSION_GALLERY
                    && shouldShowRequestPermissionRationale(Manifest.permission.WRITE_EXTERNAL_STORAGE);
            boolean shouldShowRationaleCamera = requestCode == PERMISSION_CAMERA
                    && (shouldShowRequestPermissionRationale(Manifest.permission.CAMERA)
                    /*|| shouldShowRequestPermissionRationale(Manifest.permission.WRITE_EXTERNAL_STORAGE)*/);

            final String[] permissions = requestCode == PERMISSION_CAMERA
                    ? new String[]{Manifest.permission.CAMERA/*, Manifest.permission.WRITE_EXTERNAL_STORAGE*/}
                    : new String[]{Manifest.permission.WRITE_EXTERNAL_STORAGE};

            if (shouldShowRationaleGallery || shouldShowRationaleCamera) {
                String messagePart = "";
                if (shouldShowRationaleCamera) messagePart += "camera";
//                if (shouldShowRationaleCamera && shouldShowRationaleGallery) messagePart += " and ";
                if (shouldShowRationaleGallery) messagePart += "storage";

                new AlertDialog.Builder(this)
                        .setMessage("This app requires the " + messagePart + " permission in order to scan and read MRZ")
                        .setPositiveButton("Grant", new DialogInterface.OnClickListener() {
                            @Override
                            public void onClick(DialogInterface dialog, int which) {
                                ActivityCompat.requestPermissions(PermissionsHelper.this, permissions, requestCode);
                            }
                        })
                        .setNegativeButton(android.R.string.cancel,
                                new DialogInterface.OnClickListener() {
                                    @Override
                                    public void onClick(DialogInterface dialog, int which) {
                                        showFinalWarningDialog();
                                    }
                                })
                        .create()
                        .show();
            } else {
                requestPermissions(permissions, requestCode);
            }
        }
    }

    @Override
    protected void onCreate(final Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        handleIntent(getIntent());
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        if (grantResults.length > 0) {
            boolean permissionGranted = true;

            for (int grantResult : grantResults)
                if (grantResult != PackageManager.PERMISSION_GRANTED) {
                    permissionGranted = false;
                    break;
                }

            if (permissionGranted) {
                if (callback != null) callback.onRequestedPermissionsGranted(true);
                finish();
            } else {
                showFinalWarningDialog();
            }
        }
    }

    private void showFinalWarningDialog() {
        new AlertDialog.Builder(this)
                .setMessage("The MRZ scanner cannot work without the requested permissions.")
                .setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialogInterface, int i) {
                        callback.onRequestedPermissionsGranted(false);
                        finish();
                    }
                })
                .setCancelable(false)
                .create()
                .show();
    }

}
