package com.scansolutions.mrzscannerlib;

import android.Manifest;
import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.app.DialogFragment;
import android.app.Fragment;
import android.content.Context;
import android.content.DialogInterface;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.app.ActivityCompat;
import android.view.LayoutInflater;
import android.view.SurfaceView;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.ImageButton;
import android.widget.ImageView;

public class MRZScanner extends Fragment implements Camera2Impl.CameraInitListener {

    //TODO: Handle errors;
    //TODO: Make sure no leaks occur

    private static final int REQUEST_CAMERA_PERMISSION = 1;
    private static final String FRAGMENT_DIALOG = "dialog";

    private static final int CAMERA_VERSION_1 = 1;
    private static final int CAMERA_VERSION_2 = 2;

    private static int cameraVersion = CAMERA_VERSION_2;

    private static final String TAG = "MRZCameraActivity";

    private AutoFitTextureView mTextureView;

    private ImageView debugPreview;
    private MRZOverlay mrzOverlay;

    private int x = 8;
    private int y = 20;
    private int width = 84;
    private int height = 60;

    private MRZScannerListener scannerListener;

    private Camera2Impl camera2Impl;
    private Camera1Impl camera1Impl;

    private static String dateFormat = "dd.MM.yyyy";

    @Override
    public void onResume() {
        super.onResume();

        if (cameraVersion == CAMERA_VERSION_1) {
            camera1Impl.resume();
        } else if (cameraVersion == CAMERA_VERSION_2) {
            if (mTextureView.isAvailable()) {
                camera2Impl.openCamera2(mTextureView.getWidth(), mTextureView.getHeight());
            } else {
                mTextureView.setSurfaceTextureListener(camera2Impl.mSurfaceTextureListener);
            }
        }

        Button btnCapture = getView().findViewById(R.id.captureButton);

        if (MRZCore.scannerType == ScannerType.SCANNER_TYPE_DOC_IMAGE_ID) {
            btnCapture.setVisibility(View.VISIBLE);
            btnCapture.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    MRZCore.scanningState = MRZCore.STATE_FINDING_SQUARE;
                }
            });
        } else {
            btnCapture.setVisibility(View.GONE);
        }
    }

    @Override
    public void onPause() {
        super.onPause();

        if (cameraVersion == CAMERA_VERSION_1) {
            camera1Impl.stopScanner();
        } else if (cameraVersion == CAMERA_VERSION_2) {
            camera2Impl.closeCamera2();
        }
    }

    private void requestCameraAndStoragePermission() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            if (shouldShowRequestPermissionRationale(Manifest.permission.CAMERA)
                    || shouldShowRequestPermissionRationale(Manifest.permission.WRITE_EXTERNAL_STORAGE)) {
                new ConfirmationDialog().show(getFragmentManager(), FRAGMENT_DIALOG);
            } else {
                requestPermissions(new String[]{Manifest.permission.CAMERA, Manifest.permission.WRITE_EXTERNAL_STORAGE},
                        REQUEST_CAMERA_PERMISSION);
            }
        }
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        if (requestCode == REQUEST_CAMERA_PERMISSION && grantResults.length > 0) {
            boolean permissionGranted = true;

            for (int grantResult : grantResults) {
                if (grantResult != PackageManager.PERMISSION_GRANTED) {
                    permissionGranted = false;
                }
            }

            if (!permissionGranted) {
                ErrorDialog.newInstance("The MRZ scanner cannot work without the requested permissions.")
                        .show(getFragmentManager(), FRAGMENT_DIALOG);
                return;
            }
        }

        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
    }

    static boolean hasPermissions(Context context, String... permissions) {
        if (context != null && permissions != null) {
            for (String permission : permissions) {
                if (ActivityCompat.checkSelfPermission(context, permission) != PackageManager.PERMISSION_GRANTED) {
                    return false;
                }
            }
        }
        return true;
    }

    public static class ErrorDialog extends DialogFragment {

        private static final String ARG_MESSAGE = "message";

        public static ErrorDialog newInstance(String message) {
            ErrorDialog dialog = new ErrorDialog();
            Bundle args = new Bundle();
            args.putString(ARG_MESSAGE, message);
            dialog.setArguments(args);
            return dialog;
        }

        @NonNull
        @Override
        public Dialog onCreateDialog(Bundle savedInstanceState) {
            final Activity activity = getActivity();
            return new AlertDialog.Builder(activity)
                    .setMessage(getArguments().getString(ARG_MESSAGE))
                    .setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialogInterface, int i) {
                            activity.finish();
                        }
                    })
                    .create();
        }

    }

    /**
     * Shows OK/Cancel confirmation dialog about camera permission.
     */
    public static class ConfirmationDialog extends DialogFragment {

        @NonNull
        @Override
        public Dialog onCreateDialog(Bundle savedInstanceState) {
            return new AlertDialog.Builder(getActivity())
                    .setMessage("This app requires the camera and storage permission in order to scan and read MRZ")
                    .setPositiveButton("Grant", new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                                getActivity().requestPermissions(new String[]{Manifest.permission.CAMERA, Manifest.permission.WRITE_EXTERNAL_STORAGE},
                                        REQUEST_CAMERA_PERMISSION);
                            }
                        }
                    })
                    .setNegativeButton(android.R.string.cancel,
                            new DialogInterface.OnClickListener() {
                                @Override
                                public void onClick(DialogInterface dialog, int which) {
                                    Activity activity = getActivity();
                                    if (activity != null) {
                                        activity.finish();
                                    }
                                }
                            })
                    .create();
        }
    }

    private void setScanningRectangle(int x, int y, int width, int height) {
        this.x = x;
        this.y = y;
        this.width = width;
        this.height = height;
        mrzOverlay.setRect(x, y, width, height);
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, Bundle savedInstanceState) {
        cameraVersion = (android.os.Build.VERSION.SDK_INT >= 28) ? CAMERA_VERSION_2 : CAMERA_VERSION_1;

        getActivity().getWindow().setFlags(
                WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED,
                WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED);

        new Thread(new Runnable() {
            @Override
            public void run() {
                MRZCore.prepareTesseract(getActivity());
            }
        }).start();

        View v = inflater.inflate(R.layout.amrz_activity_scanner, container, false);
        ImageButton btnFlash = v.findViewById(R.id.mrz_flash);


        mrzOverlay = v.findViewById(R.id.mrz_overlay);
        mrzOverlay.setRect(x, y, width, height);

        if (BuildConfig.BUILD_TYPE == "debug") {
            debugPreview = v.findViewById(R.id.debug_preview);
        }

        if (cameraVersion == CAMERA_VERSION_1) {
            SurfaceView mSurfaceView = v.findViewById(R.id.preview_view);
            mSurfaceView.setVisibility(View.VISIBLE);
            camera1Impl = new Camera1Impl(getActivity(), mSurfaceView, btnFlash, mrzOverlay, debugPreview, scannerListener, this);
        } else if (cameraVersion == CAMERA_VERSION_2) {
            mTextureView = v.findViewById(R.id.texture);
            mTextureView.setVisibility(View.VISIBLE);
            camera2Impl = new Camera2Impl(getActivity(), scannerListener, mTextureView, btnFlash, this, mrzOverlay, debugPreview);
        }

        return v;
    }

    @Override
    public void shouldRequestCameraPermission() {
        requestCameraAndStoragePermission();
    }

    @Override
    public void warnIncompatibleCamera() {
        new Handler(Looper.getMainLooper()).post(new Runnable() {
            @Override
            public void run() {
                new AlertDialog.Builder(getActivity())
                        .setTitle(R.string.warning)
                        .setMessage(R.string.warning_message)
                        .setNeutralButton(android.R.string.ok, null)
                        .show();
            }
        });
    }

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);

        try {
            scannerListener = (MRZScannerListener) context;
        } catch (ClassCastException e) {
            throw new ClassCastException(context.toString()
                    + "MRZScannerListener must be implemented");
        }
    }

    @Override
    public void onAttach(Activity activity) {
        super.onAttach(activity);

        try {
            scannerListener = (MRZScannerListener) activity;
        } catch (ClassCastException e) {
            throw new ClassCastException(activity.toString()
                    + "MRZScannerListener must be implemented");
        }
    }

    /**
     * Specify whether the scanner should detect and return result for IDs.
     *
     * @param isIDActive [true, false]. The default value is true.
     */
    public void setIDActive(boolean isIDActive) {
        MRZCore.isIDActive = isIDActive;
    }

    /**
     * Specify whether the scanner should detect and return result for passports.
     *
     * @param isPassportActive [true, false]. The default value is true.
     */
    public void setPassportActive(boolean isPassportActive) {
        MRZCore.isPassportActive = isPassportActive;
    }

    /**
     * Specify whether the scanner should detect and return result for visas.
     *
     * @param isVisaActive [true, false]. Default value is true.
     */
    public void setVisaActive(boolean isVisaActive) {
        MRZCore.isVisaActive = isVisaActive;
    }

    /**
     * Specify the maximum number of CPU threads that the scanner can use during the scanning process.
     *
     * @param maxThreads number of CPU threads. Default value is 2.
     */
    public void setMaxThreads(int maxThreads) {
        MRZCore.maxThreads = Math.min(MRZCore.AVAILABLE_THREADS, Math.max(maxThreads, 0));
    }

    /**
     * Specify which scanner type you want to use. There are two options: "MRZ Scanner" and "Document Image scanner".
     * The "MRZ Scanner" option is used to scan for MRZ.
     * The "Document image scanner" is used for capturing front and back image of the ID documents.
     *
     * @param scannerType [SCANNER_TYPE_MRZ, SCANNER_TYPE_DOC_IMAGE_ID, SCANNER_TYPE_DOC_IMAGE_PASSPORT]. Default value is SCANNER_TYPE_MRZ
     */
    public void setScannerType(ScannerType scannerType) {
        MRZCore.scannerType = scannerType;

        if (scannerType == ScannerType.SCANNER_TYPE_DOC_IMAGE_PASSPORT || scannerType == ScannerType.SCANNER_TYPE_DOC_IMAGE_ID) {
            setScanningRectangle(3, 25, 94, 50);
        } else {
            setScanningRectangle(8, 20, 84, 60);
        }
    }

    /**
     * Resume scanning after the scanner has been paused/stopped. Usually after a successful scan.
     */
    public void resumeScanning() {
        MRZCore.scanningState = MRZCore.STATE_SCANNING;
    }

    /**
     * Register with the licence key provided to remove the asterisks (*) from the result.
     *
     * @param key the provided licence key.
     * @return 0 for success, -1 if registration failed.
     */
    public int registerWithLicenseKey(String key) {
        String bundleID = getActivity().getApplicationInfo().packageName;
        return MRZCore.registerWithLicenseKey(key, bundleID, Build.MODEL);
    }

    /**
     * @return the current SDK Version.
     */
    public static String sdkVersion() {
        return MRZCore.sdkVersionNative();
    }

    /**
     * Set the date format in which the parsed dates are formatted.
     *
     * @param dateFormat the pattern describing the date format. Example: "dd.MM.yyyy"
     */
    public static void setDateFormat(String dateFormat) {
        MRZScanner.dateFormat = dateFormat;
    }

    /**
     * @return the currently set date format in which the parsed dates are formatted.
     */
    public static String getDateFormat() {
        return dateFormat;
    }

}
