package com.scansolutions.mrzscannerlib;

import android.Manifest;
import android.content.Context;
import android.graphics.Bitmap;
import android.hardware.Camera;
import android.os.Handler;
import android.os.Message;
import android.view.SurfaceHolder;
import android.view.SurfaceView;
import android.view.View;
import android.widget.ImageButton;
import android.widget.ImageView;

import org.opencv.android.Utils;
import org.opencv.core.CvType;
import org.opencv.core.Mat;
import org.opencv.imgproc.Imgproc;

import java.io.IOException;

import static com.scansolutions.mrzscannerlib.MRZCore.STATE_STOPPED;
import static com.scansolutions.mrzscannerlib.MRZScanner.hasPermissions;


class Camera1Impl implements SurfaceHolder.Callback {
    private boolean hasSurface;
    private CameraManager cameraManager;
    private Handler handler;
    private SurfaceHolder surfaceHolder;
    private MRZCore mrzCore;
    private Context mContext;
    private int mSensorRotation;

    private Camera2Impl.CameraInitListener cameraInitListener;

    Camera1Impl(Context context,
                SurfaceView surfaceView,
                ImageButton btnFlash,
                MRZOverlay mrzOverlay,
                ImageView debugPreview,
                MRZScannerListener scannerListener,
                Camera2Impl.CameraInitListener cameraInitListener) {
        this.cameraInitListener = cameraInitListener;
        mContext = context;
        mrzCore = new MRZCore(mrzOverlay, debugPreview, scannerListener);
        surfaceHolder = surfaceView.getHolder();
        hasSurface = false;
        cameraManager = new CameraManager(context);

        handler = new Handler(new Handler.Callback() {
            @Override
            public boolean handleMessage(Message msg) {
                if (msg.what == R.id.decode) {
                    decode((byte[]) msg.obj, msg.arg1, msg.arg2);
                }

                return false;
            }
        });

        btnFlash.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mrzCore.isTorchOn = !mrzCore.isTorchOn;
                cameraManager.setTorch(mrzCore.isTorchOn);
                if (mrzCore.isTorchOn) {
                    ((ImageButton) v).setImageResource(R.drawable.amrz_ic_flash_on);
                } else {
                    ((ImageButton) v).setImageResource(R.drawable.amrz_ic_ico_flash_off);
                }
            }
        });
    }

    void resume() {
        if (hasSurface) {
            initCamera();
        } else {
            // Install the callback and wait for surfaceCreated() to resume the
            // camera.
            surfaceHolder.addCallback(this);
            surfaceHolder.setType(SurfaceHolder.SURFACE_TYPE_PUSH_BUFFERS);
        }

        if (!mrzCore.isTorchOn) {
            cameraManager.setTorch(false);
        }
    }

    @Override
    public void surfaceCreated(SurfaceHolder holder) {
        if (!hasSurface) {
            hasSurface = true;
        }
    }

    @Override
    public void surfaceChanged(SurfaceHolder holder, int format, int width, int height) {
        initCamera();
    }

    @Override
    public void surfaceDestroyed(SurfaceHolder holder) {
        hasSurface = false;
    }

    private void initCamera() {
        if (!hasPermissions(mContext, Manifest.permission.WRITE_EXTERNAL_STORAGE, Manifest.permission.CAMERA)) {
            cameraInitListener.shouldRequestCameraPermission();
            return;
        }

        try {
            cameraManager.openDriver(surfaceHolder);
        } catch (IOException e) {
            e.printStackTrace();
        }

        //TODO: flash button
        Camera.CameraInfo cameraInfo = new Camera.CameraInfo();
        Camera.getCameraInfo(0, cameraInfo);
        mSensorRotation = cameraInfo.orientation;

        cameraManager.startPreview();
        restartPreviewAndDecode();

        if (!cameraManager.doesCameraMeetMinimum()) {
            cameraInitListener.warnIncompatibleCamera();
        }
    }

    void stopScanner() {
        MRZCore.scanningState = STATE_STOPPED;
        cameraManager.stopPreview();
        cameraManager.closeDriver();
    }

    private void restartPreviewAndDecode() {
        if (MRZCore.scanningState == STATE_STOPPED) {
            MRZCore.scanningState = MRZCore.STATE_SCANNING;
            cameraManager.requestPreviewFrame(handler, R.id.decode);
        }
    }

    private void decode(final byte[] data, final int width, final int height) {
        if (MRZCore.activeThreads >= MRZCore.maxThreads || MRZCore.scanningState == STATE_STOPPED /*|| isCameraAdjustingFocus */) {
            return;
        }

        MRZCore.activeThreads++;

        new Thread(new Runnable() {

            @Override
            public void run() {
                Mat mYuv = new Mat(height + height / 2, width, CvType.CV_8UC1);
                mYuv.put(0, 0, data);
                Mat mRgba = new Mat();
                Imgproc.cvtColor(mYuv, mRgba, Imgproc.COLOR_YUV2BGRA_NV12, 4);

                Bitmap bitmap = Bitmap.createBitmap(mRgba.cols(), mRgba.rows(), Bitmap.Config.ARGB_8888);
                Utils.matToBitmap(mRgba, bitmap);

                mRgba = mrzCore.rotateMat(mRgba, mContext, mSensorRotation);
                mrzCore.scan(mRgba);

                MRZCore.activeThreads--;
            }
        }).start();
    }
}
