package com.scansolutions.mrzscannerlib;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;
import android.view.LayoutInflater;
import android.view.SurfaceView;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.view.WindowManager;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.ScrollView;
import android.widget.Toast;

import static android.view.WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON;
import static com.scansolutions.mrzscannerlib.MRZCore.STATE_FINDING_SQUARE;
import static com.scansolutions.mrzscannerlib.MRZCore.continuousScanning;
import static com.scansolutions.mrzscannerlib.MRZCore.enableVibrator;
import static com.scansolutions.mrzscannerlib.MRZCore.getLid;
import static com.scansolutions.mrzscannerlib.MRZCore.ignoreDuplicates;
import static com.scansolutions.mrzscannerlib.MRZCore.overlayState;
import static com.scansolutions.mrzscannerlib.PermissionsHelper.PERMISSION_CAMERA;
import static com.scansolutions.mrzscannerlib.PermissionsHelper.PERMISSION_GALLERY;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_DOC_IMAGE_ID;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_ID_SESSION;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_MRZ;

public class MRZScanner extends Fragment implements CameraInitListener, MRZScannerListener {

    private static final int CAMERA_VERSION_1 = 1;
    private static final int CAMERA_VERSION_2 = 2;
    private static int cameraVersion = CAMERA_VERSION_2;
    private static String dateFormat = "dd.MM.yyyy";
    private SurfaceView mSurfaceView = null;
    private ImageView debugPreview;
    private ImageView customImageView;
    private int customImageResId = -1;
    private MRZOverlay mrzOverlay;
    private MRZScannerListener scannerListener;
    private BaseCameraImpl cameraImpl;
    private boolean permissionsWereDenied = false;
    private Bitmap customImage = null;
    private static boolean enableScanFromGallery = false;
    private ImageButton btnGallery;
    private ScannerType scannerType = SCANNER_TYPE_MRZ;
    ScrollView scrollView;
    RelativeLayout rlScrollContent;
    private ImageButton btnFlash = null;

    public MRZScanner() {
    }

    /**
     * @return the current SDK Version.
     */
    public static String sdkVersion() {
        return MRZCore.sdkVersionNative();
    }

    /**
     * @return the currently set date format in which the parsed dates are formatted.
     */
    public static String getDateFormat() {
        return dateFormat;
    }

    /**
     * Set the date format in which the parsed dates are formatted.
     *
     * @param dateFormat the pattern describing the date format. Example: "dd.MM.yyyy"
     */
    public static void setDateFormat(String dateFormat) {
        MRZScanner.dateFormat = dateFormat;
    }

    @Override
    public void onResume() {
        super.onResume();
        checkPermissionsAndInit();
    }

    private void checkPermissionsAndInit() {
        if (permissionsWereDenied) return;

        if (!PermissionsHelper.hasPermissions(getActivity(), Manifest.permission.CAMERA)) {
            PermissionsHelper.start(getActivity(), PERMISSION_CAMERA, new PermissionsHelper.PermissionsHelperCallback() {
                @Override
                public void onRequestedPermissionsGranted(boolean granted) {
                    if (granted) {
                        initScanner(true);
                    } else {
                        permissionsWereDenied = true;
                        scannerListener.permissionsWereDenied();
                    }
                }
            });
        } else {
            initScanner(false);
        }
    }

    private void initScanner(boolean forceResume) {
        cameraImpl.resume(forceResume);
    }

    @Override
    public void onPause() {
        super.onPause();
        stopScanning();
    }

    private void stopScanning() {
        cameraImpl.stopScanner();
    }

    private void pauseScanning() {
        if (!continuousScanning) {
            cameraImpl.pauseScanner();
        }
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        if (getActivity() != null) {
            getActivity().getWindow().setFlags(
                    WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED,
                    WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED);
            getActivity().getWindow().addFlags(FLAG_KEEP_SCREEN_ON);
        }
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, Bundle savedInstanceState) {
        CameraLogger.clearLogger();
        setDefaultRect();

        cameraVersion = (Build.VERSION.SDK_INT >= 21 && MRZCore.allowCamera2Support(getActivity()))
                ? CAMERA_VERSION_2
                : CAMERA_VERSION_1;

        CameraLogger.addLog("Camera version: " + cameraVersion);

        final View v = inflater.inflate(R.layout.amrz_activity_scanner, container, false);
        mrzOverlay = v.findViewById(R.id.mrz_overlay);

        initScrollContent(v);

//        if (BuildConfig.DEBUG) {
//            debugPreview = v.findViewById(R.id.debug_preview);
//        }

        initCameraPreviewImpl(v);

        if (savedInstanceState == null) {
            resetOnCleanRun();
        }

        return v;
    }

    private void resetOnCleanRun() {
        MRZCore.lastResult = "";
        if (MRZCore.overlayState == MRZOverlayState.SCANNING_BACK) {
            MRZCore.overlayState = MRZOverlayState.SCANNING_FRONT;
        }
    }

    private void setDefaultRect() {
        MRZCore.scanningRectX = 1;
        MRZCore.scanningRectY = 30;
        MRZCore.scanningRectWidth = 98;
        MRZCore.scanningRectHeight = 40;
    }

    @Override
    public void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        rlScrollContent.getViewTreeObserver().addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener() {
            @Override
            public void onGlobalLayout() {
                rlScrollContent.getViewTreeObserver().removeOnGlobalLayoutListener(this);

                setupLayoutSizes();
            }
        });
    }

    private void initScrollContent(final View v) {
        scrollView = ViewCreator.initScrollView(v);
        rlScrollContent = v.findViewById(R.id.mrz_scroll_content);
        mSurfaceView = new SurfaceView(v.getContext());

        if (customImage != null || customImageResId != -1) {
            customImageView = ViewCreator.initCustomImageView(v, rlScrollContent, customImage, customImageResId);
        }

        if (scannerType == ScannerType.SCANNER_TYPE_DOC_IMAGE_ID) {
            initCaptureButton(v);
        }

        if (scannerType == ScannerType.SCANNER_TYPE_MRZ && enableScanFromGallery) {
            createGalleryButton();
        }

        rlScrollContent.addView(mSurfaceView, 0);
        rlScrollContent.getViewTreeObserver().addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener() {
            @Override
            public void onGlobalLayout() {
                rlScrollContent.getViewTreeObserver().removeOnGlobalLayoutListener(this);

                setupLayoutSizes();
            }
        });
    }

    private void createGalleryButton() {
        if (btnGallery == null)
            btnGallery = ViewCreator.initGalleryButton(getView(), new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    pauseScanning();
                    scanFromGallery(getActivity(), scannerListener);
                }
            });
    }

    private void initCaptureButton(View v) {
        if (v != null && v.findViewById(R.id.capture_button_id) == null) {
            ViewCreator.initCaptureButton(v, new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    if ((scannerType == SCANNER_TYPE_ID_SESSION && MRZCore.overlayState == MRZOverlayState.SCANNING_FRONT)
                            || scannerType != SCANNER_TYPE_ID_SESSION) {
                        MRZCore.scanningState = STATE_FINDING_SQUARE;
                    }
                }
            });
        }
    }

    private void setupLayoutSizes() {
        if (scrollView == null || rlScrollContent == null) return;

        View v = getView();

        MRZSize viewSize = new MRZSize(v.getWidth(), v.getHeight());
        final MRZSize scaledSize = MRZUtils.scaleToDisplayRatio(MRZUtils.getDisplaySize(getActivity()), viewSize);

        MRZCore.widthRatio = v.getWidth() / scaledSize.width;
        MRZCore.heightRatio = v.getHeight() / scaledSize.height;

        ViewGroup.LayoutParams lpOverlay = mrzOverlay.getLayoutParams();
        lpOverlay.width = (int) scaledSize.width;
        lpOverlay.height = (int) scaledSize.height;
        mrzOverlay.setLayoutParams(lpOverlay);

        if (customImage == null && customImageResId == -1) {
            if (MRZUtils.isScannerTypeDoc(scannerType)) {
                mrzOverlay.setRect(1, 30, 98, 40);
            } else {
                mrzOverlay.setRect(MRZCore.scanningRectX,
                        MRZCore.scanningRectY,
                        MRZCore.scanningRectWidth,
                        MRZCore.scanningRectHeight,
                        MRZCore.widthRatio,
                        MRZCore.heightRatio);
            }
        }

        ViewGroup.LayoutParams lpScrollContent = rlScrollContent.getLayoutParams();
        lpScrollContent.width = (int) scaledSize.width;
        lpScrollContent.height = (int) scaledSize.height;
        rlScrollContent.setLayoutParams(lpScrollContent);

        mSurfaceView.setLayoutParams(new RelativeLayout.LayoutParams((int) scaledSize.width, (int) scaledSize.height));

        scrollView.getViewTreeObserver().addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener() {
            public void onGlobalLayout() {
                scrollView.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                int xPosition = (int) (scaledSize.width / 2 - scrollView.getWidth() / 2);
                int yPosition = (int) (scaledSize.height / 2 - scrollView.getHeight() / 2);
                scrollView.scrollTo(xPosition, yPosition);
            }
        });
    }

    private void initCameraPreviewImpl(View v) {
        btnFlash = v.findViewById(R.id.mrz_flash);
        if (cameraVersion == CAMERA_VERSION_1) {
            cameraImpl = new Camera1Impl(getActivity(),
                    mSurfaceView,
                    btnFlash,
                    mrzOverlay,
                    debugPreview,
                    this,
                    this,
                    scannerType);
        } else if (cameraVersion == CAMERA_VERSION_2) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                cameraImpl = new Camera2Impl(getActivity(),
                        this,
                        mSurfaceView,
                        btnFlash,
                        this,
                        mrzOverlay,
                        debugPreview,
                        scannerType);
            }
        }
    }

    @Override
    public void warnIncompatibleCamera() {
        new Handler(Looper.getMainLooper()).post(new Runnable() {
            @Override
            public void run() {
                Context c = getActivity();

                if (c != null) {
                    Toast.makeText(c, R.string.warning_message, Toast.LENGTH_LONG)
                            .show();
                }
            }
        });
    }

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);

        try {
            scannerListener = (MRZScannerListener) context;
        } catch (ClassCastException e) {
            throw new ClassCastException(context.toString()
                    + "MRZScannerListener must be implemented");
        }
    }

    @Override
    public void onAttach(Activity activity) {
        super.onAttach(activity);

        try {
            scannerListener = (MRZScannerListener) activity;
        } catch (ClassCastException e) {
            throw new ClassCastException(activity.toString()
                    + "MRZScannerListener must be implemented");
        }
    }

    @Override
    public void onDetach() {
        super.onDetach();
        mrzOverlay = null;
        if (cameraImpl != null)
            cameraImpl.release();
        mSurfaceView = null;
        scannerListener = null;
        scannerType = null;
        debugPreview = null;
        customImageView = null;
        customImage = null;
        btnGallery = null;
    }

    /**
     * Set the scanning rectangle to limit the scanning area. The parameters' values are representing percentages of the scanning preview.
     *
     * @param x      the top left point of the scanning rectangle. [0,...,100] Default value: 1.
     * @param y      the top left point of the scanning rectangle. [0,...,100] Default value: 30.
     * @param width  width the width of the scanning rectangle. [0,...,100] Default value: 98.
     * @param height height the height of the scanning rectangle. [0,...,100] Default value: 40.
     */
    public void setScanningRectangle(int x, int y, int width, int height) {
        MRZCore.scanningRectX = x;
        MRZCore.scanningRectY = y;
        MRZCore.scanningRectWidth = width;
        MRZCore.scanningRectHeight = height;

        if (customImage != null && customImageResId == -1)
            mrzOverlay.setRect(x, y, width, height, MRZCore.widthRatio, MRZCore.heightRatio);
    }

    /**
     * Specify whether the scanner should detect and return result for IDs.
     *
     * @param isIDActive [true, false]. The default value is true.
     */
    public static void setIDActive(boolean isIDActive) {
        MRZCore.isIDActive = isIDActive;
    }

    /**
     * Specify whether the scanner should detect and return result for passports.
     *
     * @param isPassportActive [true, false]. The default value is true.
     */
    public static void setPassportActive(boolean isPassportActive) {
        MRZCore.isPassportActive = isPassportActive;
    }

    /**
     * Specify whether the scanner should detect and return result for visas.
     *
     * @param isVisaActive [true, false]. Default value is true.
     */
    public static void setVisaActive(boolean isVisaActive) {
        MRZCore.isVisaActive = isVisaActive;
    }

    /**
     * Specify the maximum number of CPU threads that the scanner can use during the scanning process.
     *
     * @param maxThreads number of CPU threads. Default value is 2.
     */
    public static void setMaxThreads(int maxThreads) {
        MRZCore.maxThreads = Math.min(MRZCore.AVAILABLE_THREADS, Math.max(maxThreads, 0));
    }

    /**
     * Specify which scanner type you want to use. There are two options: "MRZ Scanner" and "Document Image scanner".
     * The "MRZ Scanner" option is used to scan for MRZ.
     * The "Document image scanner" is used for capturing front and back image of the ID documents.
     *
     * @param scannerType [SCANNER_TYPE_MRZ, SCANNER_TYPE_DOC_IMAGE_ID, SCANNER_TYPE_DOC_IMAGE_PASSPORT, SCANNER_TYPE_DOC_IMAGE_ID_FRONT]. Default value is SCANNER_TYPE_MRZ
     */
    public void setScannerType(ScannerType scannerType) {
        this.scannerType = scannerType;

        cameraImpl.setScannerType(scannerType);
        MRZCore.overlayState = MRZOverlayState.NORMAL;

        if (scannerType == SCANNER_TYPE_ID_SESSION) {
            MRZCore.overlayState = MRZOverlayState.SCANNING_FRONT;

            int orientation = getActivity().getResources().getConfiguration().orientation;
            MRZCore.setIdSessionRect(orientation);
        } else if (scannerType == SCANNER_TYPE_DOC_IMAGE_ID) {
            initCaptureButton(getView());
        }
    }

    /**
     * Resume scanning after the scanner has been paused/stopped. Usually after a successful scan when continuous scanning is not enabled.
     */
    public void resumeScanning() {
        MRZCore.lastResult = "";
        cameraImpl.resumeScanner();

        if (scannerType == SCANNER_TYPE_ID_SESSION) {
            overlayState = MRZOverlayState.SCANNING_FRONT;
            int orientation = getActivity().getResources().getConfiguration().orientation;
            MRZCore.setIdSessionRect(orientation);
        }

        if (customImage == null && customImageResId == -1) {
            if (mrzOverlay != null) {
                if (MRZUtils.isScannerTypeDoc(scannerType)) {
                    mrzOverlay.setRect(1, 30, 98, 40);
                } else {
                    mrzOverlay.setRect(MRZCore.scanningRectX,
                            MRZCore.scanningRectY,
                            MRZCore.scanningRectWidth,
                            MRZCore.scanningRectHeight,
                            MRZCore.widthRatio,
                            MRZCore.heightRatio);
                }
            }
        }

        MRZCore.scanningState = MRZCore.STATE_SCANNING;
    }

    /**
     * Register with the licence key provided to remove the asterisks (*) from the result.
     *
     * @param context                  the activity's context.
     * @param key                      the provided licence key.
     * @param mrzLicenceResultListener register result callback where int @result is 0 when registration is successful, negative number when registration failed.
     * @param mrzLicenceResultListener errorMessage is null when registration is successful, otherwise @errorMessage is an error description String.
     */
    public static void registerWithLicenseKey(final Context context, final String key, @Nullable final MRZLicenceResultListener mrzLicenceResultListener) {
        registerWithLicenseExec(context, key, mrzLicenceResultListener);
    }

    /**
     * Register with the licence key provided to remove the asterisks (*) from the result.
     *
     * @param context the activity's context.
     * @param key     the provided licence key.
     * @return 0 for success, negative number if registration failed.
     * @deprecated use {@link MRZScanner#registerWithLicenseKey(Context, String, MRZLicenceResultListener)} instead.
     */
    @Deprecated
    public static int registerWithLicenseKey(final Context context, final String key) {
        return registerWithLicenseExec(context, key, null);
    }

    private static int registerWithLicenseExec(final Context context, final String key, @Nullable final MRZLicenceResultListener mrzLicenceResultListener) {
        final String bundleID = context.getApplicationInfo().packageName;
        String lid = getLid(key);
        final int regResult = MRZCore.registerWithLicenseKey(context, key, bundleID, Build.MODEL, Build.MANUFACTURER);

        if (lid == "-2") {
            if (mrzLicenceResultListener != null) {
                mrzLicenceResultListener.onRegisterWithLicenceResult(regResult, MRZUtils.errorCodeToString(regResult));
            }
        } else {
            new Thread(new Runnable() {

                @Override
                public void run() {
                    int regReportRes = HttpsConnectionHelper.reportRegistration(context, bundleID, key, regResult);
                    final int finalRegResult = (regReportRes != MRZUtils.IGNORE_RESULT_CODE) ? regReportRes : regResult;

                    new Handler(Looper.getMainLooper()).post(new Runnable() {
                        @Override
                        public void run() {
                            if (mrzLicenceResultListener != null)
                                mrzLicenceResultListener.onRegisterWithLicenceResult(finalRegResult, MRZUtils.errorCodeToString(finalRegResult));
                        }
                    });
                }
            }).start();
        }

        return regResult;
    }

    /**
     * Scan for a single bitmap.
     *
     * @param bitmap   the image to be scanned.
     * @param activity the initialisation activity context.
     */
    public static void scanBitmap(final Bitmap bitmap, final Activity activity, final MRZScannerListener listener) {
        scanBitmapExec(bitmap, activity, listener, 16);
    }

    public static void scanBitmapReactNative(final Bitmap bitmap, final Activity activity, final MRZScannerListener listener) {
        scanBitmapExec(bitmap, activity, listener, 2);
    }

    private static void scanBitmapExec(final Bitmap bitmap, final Activity activity, final MRZScannerListener listener, int times) {
        if (!PermissionsHelper.hasPermissions(activity, Manifest.permission.READ_EXTERNAL_STORAGE)) {
            PermissionsHelper.start(activity, PERMISSION_GALLERY, new PermissionsHelper.PermissionsHelperCallback() {
                @Override
                public void onRequestedPermissionsGranted(boolean granted) {
                    if (granted)
                        scanBitmap(bitmap, activity, listener);
                    else if (listener != null)
                        listener.permissionsWereDenied();
                }
            });
            return;
        }

        new MRZCore(null, null, listener, activity, SCANNER_TYPE_MRZ)
                .scanForMRZ(bitmap, times);
    }

    /**
     * Trigger an image picker.
     *
     * @param activity the initialisation activity context.
     */
    public static void scanFromGallery(final Activity activity, final MRZScannerListener listener) {
        if (!PermissionsHelper.hasPermissions(activity, Manifest.permission.READ_EXTERNAL_STORAGE)) {
            PermissionsHelper.start(activity, PERMISSION_GALLERY, new PermissionsHelper.PermissionsHelperCallback() {
                @Override
                public void onRequestedPermissionsGranted(boolean granted) {
                    if (granted)
                        scanFromGallery(activity, listener);
                    else if (listener != null)
                        listener.permissionsWereDenied();
                }
            });
            return;
        }

        ImagePickerActivity.start(activity, listener);
    }

    /**
     * Show a warning alert dialog if the camera's resolution is lower than 5 MP and if the camera lacks the AutoFocus feature.
     *
     * @param shouldWarn [true, false]. Default value is true.
     */
    public static void warnIncompatibleCamera(boolean shouldWarn) {
        MRZCore.shouldWarnCamera = shouldWarn;
    }

    /**
     * Set valid nationality and issuing country codes. Successful scans with country codes not included in the array will be ignored.
     *
     * @param validCountryCodes array with active country codes. Example value: new String[]{"GBR", "USA"}. Default: {} (ALL).
     */
    public static void setValidCountryCodes(String[] validCountryCodes) {
        MRZCore.setValidCountryCodes(validCountryCodes);
    }

    /**
     * Specify whether validation of country codes is enabled.
     *
     * @param enabled indicates whether validation is enabled. Default: true.
     */
    public static void setValidateCountryCodesEnabled(Boolean enabled) {
        MRZCore.setValidateCountryCodesEnabled(enabled);
    }

    /**
     * Set custom overlay image for the scanner.
     *
     * @param customImage bitmap representation of the overlay image
     */
    public void setCustomImageOverlay(Bitmap customImage) {
        this.customImage = customImage;
        if (customImageView != null) {
            customImageView.setImageBitmap(customImage);
        }
    }

    /**
     * Set custom overlay image for the scanner.
     *
     * @param customImageResId resource id of the overlay image
     */
    public void setCustomImageOverlay(int customImageResId) {
        this.customImageResId = customImageResId;
        if (customImageView != null) {
            customImageView.setImageResource(customImageResId);
        }
    }

    /**
     * Enable scan from gallery. If enabled, a "scan from gallery" button will appear in the scanner preview.
     *
     * @param enabled indicates whether the "scan from gallery"  is enabled. Default value: false.
     */
    public void enableScanFromGallery(boolean enabled) {
        enableScanFromGallery = enabled;

        if (scannerType == ScannerType.SCANNER_TYPE_MRZ && enableScanFromGallery) {
            createGalleryButton();
        } else if (!enableScanFromGallery && btnGallery != null) {
            ViewGroup parent = (ViewGroup) btnGallery.getParent();
            if (parent != null) {
                parent.removeView(btnGallery);
            }
        }
    }

    /**
     * Enable upside-down scanning. If enabled, the scanner will also try the upside-down format of the preview.
     *
     * @param enabled indicates whether the "upside-down" feature  is enabled. Default value: false.
     */
    public static void setEnableUpsideDownScanning(boolean enabled) {
        MRZCore.enableUpsideDown = enabled;
    }

    /**
     * Turn flash on or off.
     *
     * @param on true = on, false = off. Default value: false.
     */
    public void toggleFlash(Boolean on) {
        cameraImpl.switchFlash(on);
    }

    /**
     * Enable vibration on successful scan.
     *
     * @param enabled indicates whether the vibration feature  is enabled. Default value: true.
     */
    public static void setEnableVibrationOnSuccess(boolean enabled) {
        enableVibrator = enabled;
    }

    /**
     * Continuous scanning. If enabled, after successful scan, the scanner view will not be paused.
     *
     * @param enabled indicates whether the scanner should stay active after a successful scan. Default value: false.
     */
    public void setContinuousScanningEnabled(boolean enabled) {
        continuousScanning = enabled;
    }

    /**
     * Ignore duplicates when scanning continuously.
     *
     * @param ignore indicates whether the scanner should repeat the last successful scan. Default value: true.
     */
    public static void setIgnoreDuplicates(boolean ignore) {
        ignoreDuplicates = ignore;
    }

    /**
     * Choose the effort level that the scanner should apply for detecting a MRZ.
     *
     * @param effortLevel available options [EFFORT_LEVEL_CASUAL, EFFORT_LEVEL_TRY_HARDER, EFFORT_LEVEL_SWEATY]. Default value: EFFORT_LEVEL_TRY_HARDER.
     */
    public static void setEffortLevel(MRZEffortLevel effortLevel) {
        MRZCore.effortLevel = effortLevel;
    }

    /**
     * Perform face detection when scanning a passport to locate the portrait image and return it as part of MRZResultModel if detected.
     * This feature is only available for Passport document types.
     *
     * @param enabled indicates whether or not to perform this action. Default value: true.
     */
    public static void setExtractPortraitEnabled(boolean enabled) {
        MRZCore.extractPortrait = enabled;
    }

    /**
     * Find and return an image of the signature as part of MRZResultModel.
     * This feature is only available for Passport document types.
     *
     * @param enabled indicates whether or not to perform this action. Default value: true.
     */
    public static void setExtractSignatureEnabled(boolean enabled) {
        MRZCore.extractSignature = enabled;
    }

    /**
     * Find and return a full image of the document as part of MRZResultModel.
     * This feature is only available for Passport document types.
     *
     * @param enabled indicates whether or not to perform this action. Default value: true.
     */
    public static void setExtractFullPassportImageEnabled(boolean enabled) {
        MRZCore.extractFullImage = enabled;
    }

    /**
     * Capture an image of the back of the ID document as part of MRZResultModel when a successful scan has been made.
     *
     * @param enabled indicates whether or not to perform this action. Default value: true.
     */
    public static void setExtractIdBackImageEnabled(boolean enabled) {
        MRZCore.extractIdBack = enabled;
    }

    /**
     * If enabled, the scanner will ignore results if check digits are not valid.
     * Disabling it increases the chances of misread.
     *
     * @param enabled default value: true.
     */
    public static void setIgnoreInvalidCheckDigitsEnabled(boolean enabled) {
        MRZCore.ignoreInvalidCheckDigits = enabled;
    }

    /**
     * Show flash button.
     *
     * @param enabled indicates whether or not to show flash button. Default value: true.
     */
    public void setShowFlashButton(boolean enabled) {
        if (btnFlash != null)
            btnFlash.setVisibility(enabled ? View.VISIBLE : View.GONE);
    }

    public void hidePreview() {
        View v = getView();
        if (v != null && v.getLayoutParams() != null) {
            v.getLayoutParams().width = 1;
            v.getLayoutParams().height = 1;
            v.setVisibility(View.INVISIBLE);
        }
    }

    @Override
    public void successfulScanWithResult(MRZResultModel resultModel) {
        scannerListener.successfulScanWithResult(resultModel);
        if (!continuousScanning)
            pauseScanning();
    }

    @Override
    public void successfulScanWithDocumentImage(Bitmap image) {
        scannerListener.successfulScanWithDocumentImage(image);
    }

    @Override
    public void successfulIdFrontImageScan(Bitmap fullImage, Bitmap portrait) {
        scannerListener.successfulIdFrontImageScan(fullImage, portrait);
    }

    @Override
    public void scanImageFailed() {
        scannerListener.scanImageFailed();
    }

    @Override
    public void permissionsWereDenied() {
        scannerListener.permissionsWereDenied();
    }

}
