package com.scansolutions.mrzscannerlib;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.ImageFormat;
import android.graphics.Point;
import android.graphics.Rect;
import android.hardware.camera2.CameraAccessException;
import android.hardware.camera2.CameraCaptureSession;
import android.hardware.camera2.CameraCharacteristics;
import android.hardware.camera2.CameraDevice;
import android.hardware.camera2.CameraManager;
import android.hardware.camera2.CameraMetadata;
import android.hardware.camera2.CaptureFailure;
import android.hardware.camera2.CaptureRequest;
import android.hardware.camera2.TotalCaptureResult;
import android.hardware.camera2.params.MeteringRectangle;
import android.hardware.camera2.params.StreamConfigurationMap;
import android.media.Image;
import android.media.ImageReader;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.RequiresApi;
import android.util.Log;
import android.util.Size;
import android.view.MotionEvent;
import android.view.Surface;
import android.view.SurfaceView;
import android.view.View;
import android.widget.ImageButton;
import android.widget.ImageView;

import org.opencv.core.CvType;
import org.opencv.core.Mat;

import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;

import static android.content.Context.CAMERA_SERVICE;
import static android.hardware.camera2.params.MeteringRectangle.METERING_WEIGHT_MAX;
import static android.media.session.PlaybackState.STATE_STOPPED;
import static com.scansolutions.mrzscannerlib.MRZCore.STATE_SCANNING;
import static com.scansolutions.mrzscannerlib.MRZCore.scanningRectHeight;
import static com.scansolutions.mrzscannerlib.MRZCore.scanningRectWidth;
import static com.scansolutions.mrzscannerlib.MRZCore.scanningRectX;
import static com.scansolutions.mrzscannerlib.MRZCore.scanningRectY;
import static com.scansolutions.mrzscannerlib.MRZCore.scanningState;

@RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
class Camera2Impl extends BaseCameraImpl {

    private boolean mManualFocusEngaged = false;

    private CameraCaptureSession mCaptureSession;
    private CameraDevice mCameraDevice;
    private ImageReader mImageReader;
    private CaptureRequest.Builder mPreviewRequestBuilder;
    private final Semaphore mCameraOpenCloseLock = new Semaphore(1);

    private boolean mFlashSupported;

    private String mCameraId;
    private Rect sensorArraySize;
    private Mat lastMat = new Mat(10, 10, CvType.CV_8UC(3));

    Camera2Impl(Context context,
                MRZScannerListener scannerListener,
                SurfaceView surfaceView,
                ImageButton btnFlash,
                CameraInitListener cameraInitListener,
                MRZOverlay mrzOverlay,
                ImageView debugPreview,
                ScannerType scannerType) {
        super(context, scannerListener, surfaceView, btnFlash, cameraInitListener, mrzOverlay, debugPreview, scannerType);

        mSurfaceView.setOnTouchListener(new View.OnTouchListener() {

            @Override
            public boolean onTouch(View view, MotionEvent motionEvent) {
                return onSurfaceTouch(view, motionEvent);
            }
        });
    }

    private boolean onSurfaceTouch(View view, MotionEvent motionEvent) {
        if (mrzCore.getScannerType() == ScannerType.SCANNER_TYPE_ID_SESSION && sensorArraySize != null) {
            final int actionMasked = motionEvent.getActionMasked();
            if (actionMasked != MotionEvent.ACTION_DOWN) {
                return false;
            }

            final int y = (int) ((motionEvent.getX() / (float) view.getWidth()) * (float) sensorArraySize.height());
            final int x = (int) ((motionEvent.getY() / (float) view.getHeight()) * (float) sensorArraySize.width());
            final int halfTouchWidth = 150; //(int)motionEvent.getTouchMajor(); //TODO: this doesn't represent actual touch size in pixel. Values range in [3, 10]...
            final int halfTouchHeight = 150; //(int)motionEvent.getTouchMinor();
            MeteringRectangle focusAreaTouch = new MeteringRectangle(Math.max(x - halfTouchWidth, 0),
                    Math.max(y - halfTouchHeight, 0),
                    halfTouchWidth * 2,
                    halfTouchHeight * 2,
                    MeteringRectangle.METERING_WEIGHT_MAX - 1);

            triggerAutoFocus(focusAreaTouch);
        }

        return true;
    }

    private CameraDevice.StateCallback mStateCallback = new CameraDevice.StateCallback() {

        @Override
        public void onOpened(@NonNull CameraDevice cameraDevice) {
            CameraLogger.addLog("onOpened");

            // This method is called when the camera is opened.  We start camera preview here.
            mCameraOpenCloseLock.release();
            mCameraDevice = cameraDevice;
            createCamera2PreviewSession();
        }

        @Override
        public void onDisconnected(@NonNull CameraDevice cameraDevice) {
            CameraLogger.addLog("onDisconnected");

            mCameraOpenCloseLock.release();
            cameraDevice.close();
            mCameraDevice = null;
        }

        @Override
        public void onError(@NonNull CameraDevice cameraDevice, int error) {
            CameraLogger.addLog("onError: " + error);

            mCameraOpenCloseLock.release();
            cameraDevice.close();
            mCameraDevice = null;
//            MRZScanner.this.finish(); TODO: Handle this
        }

    };

    private ImageReader.OnImageAvailableListener mOnImageAvailableListener = new ImageReader.OnImageAvailableListener() {

        @Override
        public void onImageAvailable(ImageReader reader) {
            final Image img = reader.acquireLatestImage();

            if (MRZCore.activeThreads >= MRZCore.maxThreads || MRZCore.scanningState == STATE_STOPPED /*|| isCameraAdjustingFocus */ || img == null) {
                if (img != null) {
                    img.close();
                }

                return;
            }

            MRZCore.activeThreads++;

            new Thread(new Runnable() {

                @Override
                public void run() {
                    try {
                        Mat imageGrab = ImageUtils.imageToMat(img);
                        img.close();

                        ImageUtils.rotateMat(imageGrab, MRZUtils.getRotation(mContext), mSensorOrientation);

                        if (mrzCore.getScannerType() == ScannerType.SCANNER_TYPE_ID_SESSION &&
                                MRZCore.checkForMotion(imageGrab.getNativeObjAddr(),
                                        lastMat.getNativeObjAddr(),
                                        scanningRectX,
                                        scanningRectY,
                                        scanningRectWidth,
                                        scanningRectHeight)) {
                            triggerAutoFocus(null);
                        }

                        boolean isPreview1x1 = mSurfaceView.getWidth() == 1 && mSurfaceView.getHeight() == 1;
                        mrzCore.scan(imageGrab, isPreview1x1);
//                        imageGrab.release();
                    } catch (IllegalStateException ignored) {
                    }

                    MRZCore.activeThreads--;
                }
            }).start();
        }
    };

    private void triggerAutoFocus(final MeteringRectangle focusAreaTouch) {
        new Handler(Looper.getMainLooper()).post(new Runnable() {
            @Override
            public void run() {
                if (mCaptureSession != null) {
                    try {
                        if (mManualFocusEngaged) {
                            return;
                        }

                        CameraCaptureSession.CaptureCallback captureCallbackHandler = new CameraCaptureSession.CaptureCallback() {
                            @Override
                            public void onCaptureCompleted(CameraCaptureSession session, CaptureRequest request, TotalCaptureResult result) {
                                super.onCaptureCompleted(session, request, result);
                                mManualFocusEngaged = false;

                                if (mCaptureSession != null) {
                                    if (request.getTag() == "FOCUS_TAG") {
                                        //the focus trigger is complete -
                                        //resume repeating (preview surface will get frames), clear AF trigger
                                        mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AF_TRIGGER, null);
                                        try {
                                            mCaptureSession.setRepeatingRequest(mPreviewRequestBuilder.build(), null, null);
                                        } catch (CameraAccessException e) {
                                            e.printStackTrace();
                                        }
                                    }
                                }
                            }

                            @Override
                            public void onCaptureFailed(CameraCaptureSession session, CaptureRequest request, CaptureFailure failure) {
                                super.onCaptureFailed(session, request, failure);
                                Log.e(TAG, "Manual AF failure: " + failure);
                                mManualFocusEngaged = false;
                            }
                        };

                        mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AF_TRIGGER, CameraMetadata.CONTROL_AF_TRIGGER_CANCEL);
                        mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AF_MODE, CaptureRequest.CONTROL_AF_MODE_OFF);
                        mCaptureSession.capture(mPreviewRequestBuilder.build(), captureCallbackHandler, null);
                        MeteringRectangle focusAreaTouch2 = getMeteringRectangle();

                        CameraCharacteristics characteristics = ((CameraManager) mContext.getSystemService(CAMERA_SERVICE))
                                .getCameraCharacteristics(mCameraId);

                        if (characteristics.get(CameraCharacteristics.CONTROL_MAX_REGIONS_AF) >= 1) {
                            mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AF_REGIONS, new MeteringRectangle[]{
                                    focusAreaTouch != null
                                            ? focusAreaTouch
                                            : focusAreaTouch2
                            });
                        }
                        if (characteristics.get(CameraCharacteristics.CONTROL_MAX_REGIONS_AE) >= 1) {
                            mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AE_REGIONS, new MeteringRectangle[]{
                                    focusAreaTouch != null
                                            ? focusAreaTouch
                                            : focusAreaTouch2
                            });
                        }

                        mPreviewRequestBuilder.set(CaptureRequest.CONTROL_MODE, CameraMetadata.CONTROL_MODE_AUTO);
                        mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AF_MODE, CaptureRequest.CONTROL_AF_MODE_AUTO);
                        mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AF_TRIGGER, CameraMetadata.CONTROL_AF_TRIGGER_START);
                        mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AE_PRECAPTURE_TRIGGER, CameraMetadata.CONTROL_AE_MODE_ON);
                        mPreviewRequestBuilder.setTag("FOCUS_TAG"); //we'll capture this later for resuming the preview

                        mCaptureSession.capture(mPreviewRequestBuilder.build(), captureCallbackHandler, null);
                        mManualFocusEngaged = true;
                    } catch (CameraAccessException e) {
                        e.printStackTrace();
                    }
                }
            }
        });
    }

    private MeteringRectangle getMeteringRectangle() {
        MeteringRectangle meteringRectangle;
        int orientation = (ImageUtils.ORIENTATIONS.get(MRZUtils.getRotation(mContext)) + mSensorOrientation + 270) % 360;

        switch (orientation) {
            case 90:
                meteringRectangle = generateMeteringRect(scanningRectY, scanningRectX, scanningRectHeight, scanningRectWidth);
                break;
            case 180:
                meteringRectangle = generateMeteringRect((100 - scanningRectX - scanningRectWidth), (100 - MRZCore.scanningRectY - scanningRectHeight),
                        scanningRectWidth, scanningRectHeight);
                break;
            case 270:
                meteringRectangle = generateMeteringRect((100 - MRZCore.scanningRectY - scanningRectHeight), (100 - scanningRectX - scanningRectWidth),
                        scanningRectHeight, scanningRectWidth);
                break;
            default:
                meteringRectangle = generateMeteringRect(scanningRectX, scanningRectY, scanningRectWidth, scanningRectHeight);
        }
        return meteringRectangle;
    }

    private MeteringRectangle generateMeteringRect(float x, float y, float width, float height) {
        return new MeteringRectangle(
                (int) (x / 100.0f * (float) sensorArraySize.width()),
                (int) (y / 100.0f * (float) sensorArraySize.height()),
                (int) (width / 100.0f * (float) sensorArraySize.width()),
                (int) (height / 100.0f * (float) sensorArraySize.height()),
                METERING_WEIGHT_MAX
        );
    }

    private boolean setUpCameraOutputs() throws NullPointerException {
        boolean afSupported = false;
        boolean resSupported = false;

        CameraManager manager = (CameraManager) mContext.getSystemService(CAMERA_SERVICE);
        try {
            CameraLogger.addLog("Available CameraIDs: ");
            for (String cameraId : manager.getCameraIdList()) {
                CameraLogger.addLog(cameraId);
                CameraCharacteristics characteristics = manager.getCameraCharacteristics(cameraId);

                StreamConfigurationMap map = characteristics.get(CameraCharacteristics.SCALER_STREAM_CONFIGURATION_MAP);
                if (map == null) {
                    CameraLogger.addLog("map == null");
                    continue;
                }
                CameraLogger.addLog(map.toString());
            }

            for (String cameraId : manager.getCameraIdList()) {
                CameraLogger.addLog("Camera ID " + cameraId);

                CameraCharacteristics characteristics = manager.getCameraCharacteristics(cameraId);

                // We don't use the front facing camera
                Integer facing = characteristics.get(CameraCharacteristics.LENS_FACING);
                if (facing != null && facing == CameraCharacteristics.LENS_FACING_FRONT) {
                    CameraLogger.addLog("CameraCharacteristics.LENS_FACING_FRONT");
                    continue;
                }

                StreamConfigurationMap map = characteristics.get(CameraCharacteristics.SCALER_STREAM_CONFIGURATION_MAP);
                if (map == null) {
                    CameraLogger.addLog("map == null");
                    continue;
                }

                if (!MRZUtils.contains(map.getOutputFormats(), ImageFormat.YUV_420_888)) {
                    CameraLogger.addLog("YUV_420_888 not supported");
                    continue;
                }

                Point displaySize = MRZUtils.getDisplaySize(mContext);
                int maxPreviewWidth = displaySize.x;
                int maxPreviewHeight = displaySize.y;

                CameraLogger.addLog("displaySize " + maxPreviewWidth + " x " + maxPreviewHeight);

                Size[] sizes = map.getOutputSizes(ImageFormat.YUV_420_888);

                List<MRZSize> mappedSizes = MRZUtils.mapSizes(sizes);
                Point bestFit = MRZUtils.getBestFitSize(mappedSizes, Math.max(maxPreviewWidth, maxPreviewHeight) / (float) Math.min(maxPreviewWidth, maxPreviewHeight));

                for (Size size : sizes)
                    if (size.getWidth() * size.getHeight() > 4500000) {
                        resSupported = true;
                        break;
                    }

                int[] afModes = characteristics.get(CameraCharacteristics.CONTROL_AF_AVAILABLE_MODES);
                if (afModes != null && afModes.length > 1) {
                    afSupported = true;

                    CameraLogger.addLog("AF Modes: ");

                    for (int afMode : afModes) {
                        CameraLogger.addLog("" + afMode);
                    }
                }

                mImageReader = ImageReader.newInstance(bestFit.x, bestFit.y, ImageFormat.YUV_420_888, MRZCore.maxThreads + 2);
                mImageReader.setOnImageAvailableListener(mOnImageAvailableListener, null);
                mSurfaceView.getHolder().setFixedSize(bestFit.x, bestFit.y);

                //noinspection ConstantConditions
                mSensorOrientation = characteristics.get(CameraCharacteristics.SENSOR_ORIENTATION);

                // Check if the flash is supported.
                Boolean available = characteristics.get(CameraCharacteristics.FLASH_INFO_AVAILABLE);
                mFlashSupported = available == null ? false : available;
                mCameraId = cameraId;

                CameraLogger.addLog("Flash supported: " + mFlashSupported);

                if (!mFlashSupported) {
                    btnFlash.setVisibility(View.GONE);
                }

                if (afSupported && resSupported) {
                    CameraLogger.addLog("Chosen camera id : " + cameraId);

                    sensorArraySize = characteristics.get(CameraCharacteristics.SENSOR_INFO_ACTIVE_ARRAY_SIZE);
                    return true;
                }
            }
        } catch (CameraAccessException e) {
            e.printStackTrace();
        }

        return false;
    }

    private boolean cameraOpened = false;

    @SuppressLint("MissingPermission")
    @Override
    void initCamera() {
        if (!cameraOpened) {
            cameraOpened = true;

            boolean doesCameraMeetRequirements = setUpCameraOutputs();

            CameraManager manager = (CameraManager) mContext.getSystemService(CAMERA_SERVICE);
            try {
                if (!mCameraOpenCloseLock.tryAcquire(2500, TimeUnit.MILLISECONDS)) return;

                manager.openCamera(mCameraId, mStateCallback, null);
            } catch (CameraAccessException e) {
                CameraLogger.addLog("openCamera2 Error : " + e.getMessage());

                e.printStackTrace();
            } catch (InterruptedException e) {
                CameraLogger.addLog("openCamera2 Error : " + e.getMessage());

                throw new RuntimeException("Interrupted while trying to lock camera opening.", e);
            }

            if (!doesCameraMeetRequirements && MRZCore.shouldWarnCamera) {
                cameraInitListener.warnIncompatibleCamera();
            }
        }
    }

    @Override
    void pauseScanner() {
        try {
            if (null != mCaptureSession) {
                mCaptureSession.stopRepeating();
            }
        } catch (CameraAccessException | IllegalStateException e) {
            e.printStackTrace();
        }
    }

    @Override
    void resumeScanner() {
        if (mCaptureSession != null) {
            try {
                if (mrzCore.isTorchOn) {
                    mPreviewRequestBuilder.set(CaptureRequest.FLASH_MODE, CaptureRequest.FLASH_MODE_TORCH);
                }

                mCaptureSession.setRepeatingRequest(mPreviewRequestBuilder.build(), null, null);
            } catch (CameraAccessException e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    void stopScanner() {
        cameraOpened = false;
        scanningState = STATE_STOPPED;

        try {
            mCameraOpenCloseLock.acquire();
            if (null != mCaptureSession) {
                mCaptureSession.close();
                mCaptureSession = null;
            }
            if (null != mCameraDevice) {
                mCameraDevice.close();
                mCameraDevice = null;
            }
            if (null != mImageReader) {
                mImageReader.close();
                mImageReader = null;
            }
        } catch (InterruptedException e) {
            throw new RuntimeException("Interrupted while trying to lock camera closing.", e);
        } finally {
            mCameraOpenCloseLock.release();
        }
    }

    private void createCamera2PreviewSession() {
        try {
            Surface surface = mSurfaceView.getHolder().getSurface();

            if (surface == null || mImageReader == null)
                return;

            // We set up a CaptureRequest.Builder with the output Surface.
            mPreviewRequestBuilder = mCameraDevice.createCaptureRequest(CameraDevice.TEMPLATE_PREVIEW);
            mPreviewRequestBuilder.addTarget(surface);
            mPreviewRequestBuilder.addTarget(mImageReader.getSurface());

            // Here, we create a CameraCaptureSession for camera preview.
            mCameraDevice.createCaptureSession(Arrays.asList(surface, mImageReader.getSurface()), new CameraCaptureSession.StateCallback() {

                @Override
                public void onConfigured(@NonNull CameraCaptureSession cameraCaptureSession) {
                    // The camera is already closed
                    if (null == mCameraDevice) {
                        return;
                    }

                    // When the session is ready, we start displaying the preview.
                    mCaptureSession = cameraCaptureSession;
                    try {
                        if (mrzCore.getScannerType() == ScannerType.SCANNER_TYPE_ID_SESSION) {
                            triggerAutoFocus(null);
                        } else {
                            mPreviewRequestBuilder.set(CaptureRequest.CONTROL_AF_MODE, CaptureRequest.CONTROL_AF_MODE_CONTINUOUS_PICTURE);
                        }

                        if (mFlashSupported && mrzCore.isTorchOn) {
                            mPreviewRequestBuilder.set(CaptureRequest.FLASH_MODE, CaptureRequest.FLASH_MODE_TORCH);
                            btnFlash.setImageResource(R.drawable.amrz_ic_flash_on);
                        }

                        mCaptureSession.setRepeatingRequest(mPreviewRequestBuilder.build(), null, null);
                        MRZCore.scanningState = STATE_SCANNING;
                    } catch (CameraAccessException e) {
                        CameraLogger.addLog("onConfigured Error : " + e.getMessage());

                        e.printStackTrace();
                    } catch (IllegalStateException e) {
                        CameraLogger.addLog("onConfigured Error : " + e.getMessage());

                        e.printStackTrace();
                    }
                }

                @Override
                public void onConfigureFailed(@NonNull CameraCaptureSession cameraCaptureSession) {
                    CameraLogger.addLog("onConfigureFailed");
                    Log.i(TAG, "onConfigureFailed");
                }
            }, null);
        } catch (CameraAccessException e) {
            CameraLogger.addLog("CameraAccessException " + e.getMessage());

            e.printStackTrace();
        }
    }

    @Override
    void switchFlash(Boolean flashState) {
        mrzCore.isTorchOn = flashState;

        if (mFlashSupported && mCaptureSession != null) {
            try {
                if (flashState) {
                    mPreviewRequestBuilder.set(CaptureRequest.FLASH_MODE, CaptureRequest.FLASH_MODE_TORCH);
                    btnFlash.setImageResource(R.drawable.amrz_ic_flash_on);
                } else {
                    mPreviewRequestBuilder.set(CaptureRequest.FLASH_MODE, CaptureRequest.FLASH_MODE_OFF);
                    btnFlash.setImageResource(R.drawable.amrz_ic_ico_flash_off);
                }

                mCaptureSession.setRepeatingRequest(mPreviewRequestBuilder.build(), null, null);
            } catch (CameraAccessException e) {
                e.printStackTrace();
            } catch (IllegalStateException e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    void release() {
        cameraInitListener = null;
        mCaptureSession = null;
        mCameraDevice = null;
        mImageReader = null;
        mPreviewRequestBuilder = null;
        mContext = null;
        mCameraId = null;
        mrzCore = null;
        mSurfaceView = null;
        btnFlash = null;
        sensorArraySize = null;
        lastMat = null;
        mOnImageAvailableListener = null;
        mStateCallback = null;
    }
}
