package com.samebug.notifier;

import java.lang.Thread.UncaughtExceptionHandler;
import java.util.Date;
import java.util.UUID;

import com.samebug.notifier.exceptions.NotifierException;
import com.samebug.notifier.proxy.ThrowableProxy;

public class Samebug {

	/**
	 * Install a default exception handler on the current thread, configured by
	 * the property file named 'samebug.properties' in the classpath.
	 * 
	 * In case of any error (properties file not found, multiple file found,
	 * wrong format of properties, missing obligatory entries) the constructed
	 * notifier will not function as intended.
	 */
	public static void handleUncaughtExceptions() {
		SamebugUncaughtExceptionHandler handler = new SamebugUncaughtExceptionHandler();
		installHandler(handler);
	}

	/**
	 * Install a default exception handler on the current thread, with the given
	 * application key.
	 */
	public static void handleUncaughtExceptions(final String applicationKey) {
		SamebugUncaughtExceptionHandler handler = new SamebugUncaughtExceptionHandler(applicationKey);
		installHandler(handler);
	}

	private static void installHandler(final UncaughtExceptionHandler handler) {
		Thread.setDefaultUncaughtExceptionHandler(handler);
	}

	private static boolean initialized = false;
	private static INotifier notifier;

	public static synchronized void init() {
		if (!initialized) {
			notifier = new SamebugNotifier();
			initialized = true;
		}
	}

	public static synchronized void init(final String applicationKey) {
		if (!initialized) {
			notifier = new SamebugNotifier(applicationKey);
			initialized = true;
		}
	}

	/**
	 * Notify Samebug about an exception
	 * 
	 * @param message custom message describing the event
	 * @param throwable the caught exception
	 * @return A UUID of the bug where this report was categorized
	 * @throws NotifierException
	 */
	public static UUID notify(final String message, final Throwable throwable) throws NotifierException {
		if (!initialized) {
			throw new IllegalArgumentException("Samebug not initialized");
		} else {
			return notifier.notify(message, throwable);
		}
	}

	/**
	 * Notify Samebug about an exception
	 * 
	 * @param message custom message describing the event
	 * @param throwable the caught exception
	 * @param timestamp the time of occurrence
	 * @return A UUID of the bug where this report was categorized
	 * @throws NotifierException
	 */
	public static UUID notify(final String message, final Throwable throwable, final Date timestamp) throws NotifierException {
		if (!initialized) {
			throw new IllegalArgumentException("Samebug not initialized");
		} else {
			return notifier.notify(message, throwable, timestamp);
		}
	}

	/**
	 * Notify Samebug about an exception
	 * 
	 * @param message custom message describing the event
	 * @param throwableProxy proxy for the caught exception
	 * @return A UUID of the bug where this report was categorized
	 * @throws NotifierException
	 */
	public static UUID notify(final String message, final ThrowableProxy throwableProxy) throws NotifierException {
		if (!initialized) {
			throw new IllegalArgumentException("Samebug not initialized");
		} else {
			return notifier.notify(message, throwableProxy);
		}
	}

	/**
	 * Notify Samebug about an exception
	 * 
	 * @param message custom message describing the event
	 * @param throwableProxy proxy for the caught exception
	 * @param timestamp the time of occurrence
	 * @return A UUID of the bug where this report was categorized
	 * @throws NotifierException
	 */
	public static UUID notify(final String message, final ThrowableProxy throwableProxy, final Date timestamp) throws NotifierException {
		if (!initialized) {
			throw new IllegalArgumentException("Samebug not initialized");
		} else {
			return notifier.notify(message, throwableProxy, timestamp);
		}
	}
}
