/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.internal;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import io.grpc.Attributes;
import io.grpc.ChannelLogger;
import io.grpc.ConnectivityState;
import io.grpc.ConnectivityStateInfo;
import io.grpc.EquivalentAddressGroup;
import io.grpc.LoadBalancer;
import io.grpc.LoadBalancerProvider;
import io.grpc.LoadBalancerRegistry;
import io.grpc.Status;
import io.grpc.internal.GrpcAttributes;
import io.grpc.internal.ServiceConfigUtil;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import javax.annotation.Nullable;

public final class AutoConfiguredLoadBalancerFactory
extends LoadBalancer.Factory {
    private static final Logger logger = Logger.getLogger(AutoConfiguredLoadBalancerFactory.class.getName());
    private final LoadBalancerRegistry registry;
    private final String defaultPolicy;

    public AutoConfiguredLoadBalancerFactory(String defaultPolicy) {
        this(LoadBalancerRegistry.getDefaultRegistry(), defaultPolicy);
    }

    @VisibleForTesting
    AutoConfiguredLoadBalancerFactory(LoadBalancerRegistry registry, String defaultPolicy) {
        this.registry = Preconditions.checkNotNull(registry, "registry");
        this.defaultPolicy = Preconditions.checkNotNull(defaultPolicy, "defaultPolicy");
    }

    @Override
    public LoadBalancer newLoadBalancer(LoadBalancer.Helper helper) {
        return new AutoConfiguredLoadBalancer(helper);
    }

    private LoadBalancerProvider getProviderOrThrow(String policy, String choiceReason) throws PolicyException {
        LoadBalancerProvider provider = this.registry.getProvider(policy);
        if (provider == null) {
            throw new PolicyException("Trying to load '" + policy + "' because " + choiceReason + ", but it's unavailable");
        }
        return provider;
    }

    private static final class FailingPicker
    extends LoadBalancer.SubchannelPicker {
        private final Status failure;

        FailingPicker(Status failure) {
            this.failure = failure;
        }

        @Override
        public LoadBalancer.PickResult pickSubchannel(LoadBalancer.PickSubchannelArgs args) {
            return LoadBalancer.PickResult.withError(this.failure);
        }
    }

    private static final class EmptyPicker
    extends LoadBalancer.SubchannelPicker {
        private EmptyPicker() {
        }

        @Override
        public LoadBalancer.PickResult pickSubchannel(LoadBalancer.PickSubchannelArgs args) {
            return LoadBalancer.PickResult.withNoResult();
        }
    }

    @VisibleForTesting
    static final class PolicySelection {
        final LoadBalancerProvider provider;
        final List<EquivalentAddressGroup> serverList;
        @Nullable
        final Map<String, Object> config;

        PolicySelection(LoadBalancerProvider provider, List<EquivalentAddressGroup> serverList, @Nullable Map<?, ?> config) {
            this.provider = Preconditions.checkNotNull(provider, "provider");
            this.serverList = Collections.unmodifiableList(Preconditions.checkNotNull(serverList, "serverList"));
            this.config = config;
        }
    }

    @VisibleForTesting
    static class PolicyException
    extends Exception {
        private static final long serialVersionUID = 1L;

        private PolicyException(String msg) {
            super(msg);
        }
    }

    @VisibleForTesting
    public final class AutoConfiguredLoadBalancer
    extends LoadBalancer {
        private final LoadBalancer.Helper helper;
        private LoadBalancer delegate;
        private LoadBalancerProvider delegateProvider;
        private boolean roundRobinDueToGrpclbDepMissing;

        AutoConfiguredLoadBalancer(LoadBalancer.Helper helper) {
            this.helper = helper;
            this.delegateProvider = AutoConfiguredLoadBalancerFactory.this.registry.getProvider(AutoConfiguredLoadBalancerFactory.this.defaultPolicy);
            if (this.delegateProvider == null) {
                throw new IllegalStateException("Could not find policy '" + AutoConfiguredLoadBalancerFactory.this.defaultPolicy + "'. Make sure its implementation is either registered to LoadBalancerRegistry or included in META-INF/services/io.grpc.LoadBalancerProvider from your jar files.");
            }
            this.delegate = this.delegateProvider.newLoadBalancer(helper);
        }

        @Override
        public void handleResolvedAddressGroups(List<EquivalentAddressGroup> servers, Attributes attributes) {
            PolicySelection selection;
            if (attributes.get(ATTR_LOAD_BALANCING_CONFIG) != null) {
                throw new IllegalArgumentException("Unexpected ATTR_LOAD_BALANCING_CONFIG from upstream: " + attributes.get(ATTR_LOAD_BALANCING_CONFIG));
            }
            Map<String, Object> configMap = attributes.get(GrpcAttributes.NAME_RESOLVER_SERVICE_CONFIG);
            try {
                selection = this.decideLoadBalancerProvider(servers, configMap);
            }
            catch (PolicyException e) {
                Status s = Status.INTERNAL.withDescription(e.getMessage());
                this.helper.updateBalancingState(ConnectivityState.TRANSIENT_FAILURE, new FailingPicker(s));
                this.delegate.shutdown();
                this.delegateProvider = null;
                this.delegate = new NoopLoadBalancer();
                return;
            }
            if (this.delegateProvider == null || !selection.provider.getPolicyName().equals(this.delegateProvider.getPolicyName())) {
                this.helper.updateBalancingState(ConnectivityState.CONNECTING, new EmptyPicker());
                this.delegate.shutdown();
                this.delegateProvider = selection.provider;
                LoadBalancer old = this.delegate;
                this.delegate = this.delegateProvider.newLoadBalancer(this.helper);
                this.helper.getChannelLogger().log(ChannelLogger.ChannelLogLevel.INFO, "Load balancer changed from {0} to {1}", old.getClass().getSimpleName(), this.delegate.getClass().getSimpleName());
            }
            if (selection.config != null) {
                this.helper.getChannelLogger().log(ChannelLogger.ChannelLogLevel.DEBUG, "Load-balancing config: {0}", selection.config);
                attributes = attributes.toBuilder().set(ATTR_LOAD_BALANCING_CONFIG, selection.config).build();
            }
            LoadBalancer delegate = this.getDelegate();
            if (selection.serverList.isEmpty() && !delegate.canHandleEmptyAddressListFromNameResolution()) {
                delegate.handleNameResolutionError(Status.UNAVAILABLE.withDescription("Name resolver returned no usable address. addrs=" + servers + ", attrs=" + attributes));
            } else {
                delegate.handleResolvedAddressGroups(selection.serverList, attributes);
            }
        }

        @Override
        public void handleNameResolutionError(Status error) {
            this.getDelegate().handleNameResolutionError(error);
        }

        @Override
        public void handleSubchannelState(LoadBalancer.Subchannel subchannel, ConnectivityStateInfo stateInfo) {
            this.getDelegate().handleSubchannelState(subchannel, stateInfo);
        }

        @Override
        public boolean canHandleEmptyAddressListFromNameResolution() {
            return true;
        }

        @Override
        public void shutdown() {
            this.delegate.shutdown();
            this.delegate = null;
        }

        @VisibleForTesting
        public LoadBalancer getDelegate() {
            return this.delegate;
        }

        @VisibleForTesting
        void setDelegate(LoadBalancer lb) {
            this.delegate = lb;
        }

        @VisibleForTesting
        LoadBalancerProvider getDelegateProvider() {
            return this.delegateProvider;
        }

        @VisibleForTesting
        PolicySelection decideLoadBalancerProvider(List<EquivalentAddressGroup> servers, @Nullable Map<String, Object> config) throws PolicyException {
            boolean haveBalancerAddress = false;
            ArrayList<EquivalentAddressGroup> backendAddrs = new ArrayList<EquivalentAddressGroup>();
            for (EquivalentAddressGroup s : servers) {
                if (s.getAttributes().get(GrpcAttributes.ATTR_LB_ADDR_AUTHORITY) != null) {
                    haveBalancerAddress = true;
                    continue;
                }
                backendAddrs.add(s);
            }
            if (haveBalancerAddress) {
                LoadBalancerProvider grpclbProvider = AutoConfiguredLoadBalancerFactory.this.registry.getProvider("grpclb");
                if (grpclbProvider == null) {
                    if (backendAddrs.isEmpty()) {
                        throw new PolicyException("Received ONLY balancer addresses but grpclb runtime is missing");
                    }
                    if (!this.roundRobinDueToGrpclbDepMissing) {
                        this.roundRobinDueToGrpclbDepMissing = true;
                        String errorMsg = "Found balancer addresses but grpclb runtime is missing. Will use round_robin. Please include grpc-grpclb in your runtime depedencies.";
                        this.helper.getChannelLogger().log(ChannelLogger.ChannelLogLevel.ERROR, errorMsg);
                        logger.warning(errorMsg);
                    }
                    return new PolicySelection(AutoConfiguredLoadBalancerFactory.this.getProviderOrThrow("round_robin", "received balancer addresses but grpclb runtime is missing"), backendAddrs, null);
                }
                return new PolicySelection(grpclbProvider, servers, null);
            }
            this.roundRobinDueToGrpclbDepMissing = false;
            List<Map<String, Object>> lbConfigs = null;
            if (config != null) {
                lbConfigs = ServiceConfigUtil.getLoadBalancingConfigsFromServiceConfig(config);
            }
            if (lbConfigs != null && !lbConfigs.isEmpty()) {
                LinkedHashSet<String> policiesTried = new LinkedHashSet<String>();
                for (Map<String, Object> lbConfig : lbConfigs) {
                    if (lbConfig.size() != 1) {
                        throw new PolicyException("There are " + lbConfig.size() + " load-balancing configs in a list item. Exactly one is expected. Config=" + lbConfig);
                    }
                    Map.Entry<String, Object> entry = lbConfig.entrySet().iterator().next();
                    String policy = entry.getKey();
                    LoadBalancerProvider provider = AutoConfiguredLoadBalancerFactory.this.registry.getProvider(policy);
                    if (provider != null) {
                        if (!policiesTried.isEmpty()) {
                            this.helper.getChannelLogger().log(ChannelLogger.ChannelLogLevel.DEBUG, "{0} specified by Service Config are not available", policiesTried);
                        }
                        return new PolicySelection(provider, servers, (Map)entry.getValue());
                    }
                    policiesTried.add(policy);
                }
                throw new PolicyException("None of " + policiesTried + " specified by Service Config are available.");
            }
            return new PolicySelection(AutoConfiguredLoadBalancerFactory.this.getProviderOrThrow(AutoConfiguredLoadBalancerFactory.this.defaultPolicy, "using default policy"), servers, null);
        }
    }

    private static final class NoopLoadBalancer
    extends LoadBalancer {
        private NoopLoadBalancer() {
        }

        @Override
        public void handleResolvedAddressGroups(List<EquivalentAddressGroup> s, Attributes a) {
        }

        @Override
        public void handleNameResolutionError(Status error) {
        }

        @Override
        public void handleSubchannelState(LoadBalancer.Subchannel subchannel, ConnectivityStateInfo stateInfo) {
        }

        @Override
        public void shutdown() {
        }
    }
}

