package com.rmkrishna.mediapicker

import android.net.Uri
import androidx.appcompat.app.AppCompatActivity
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentActivity
import com.rmkrishna.permission.MPermission
import com.rmkrishna.permissionX.askPermissions

private const val MFragment_TAG = "MFragment_TAG"

/**
 * To check the permission from AppCompatActivity
 */
fun AppCompatActivity.getImage(
    listener: PickerListener.() -> Unit
) {
    val mediaPickerListener: MediaPickerListener =
        getPickerListener(listener)
    askPermissions(
        MPermission.WRITE_EXTERNAL_STORAGE,
        MPermission.READ_EXTERNAL_STORAGE,
        MPermission.CAMERA
    ) {
        granted { getMedia(mediaPickerListener) }
        denied { mediaPickerListener.noAccess() }
    }

}

/**
 * To check the permission from Fragment
 */
fun Fragment.getImage(
    listener: PickerListener.() -> Unit
) {
    val mediaPickerListener: MediaPickerListener =
        getPickerListener(listener)

    askPermissions(
        MPermission.WRITE_EXTERNAL_STORAGE,
        MPermission.READ_EXTERNAL_STORAGE,
        MPermission.CAMERA
    ) {
        granted {
            activity?.let {
                it.getMedia(
                    mediaPickerListener
                )
            }
        }
        denied { mediaPickerListener.noAccess() }
    }
}

/**
 * To check the permission from FragmentActivity
 */
private fun FragmentActivity.getMedia(
    listener: MediaPickerListener
) {
    val mFragment = supportFragmentManager.findFragmentByTag(MFragment_TAG)

    if (mFragment == null) {
        var fragment = MFragment()

        fragment = fragment.setListener(listener)

        supportFragmentManager.beginTransaction().add(
            fragment,
            MFragment_TAG
        )
            .commitAllowingStateLoss()
    }
}


interface MediaPickerListener {
    fun onFileReceived(files: List<Uri?>)
    fun noAccess()
    fun onFileReceivedFailed()
}

fun getPickerListener(listener: PickerListener.() -> Unit) =
    PickerListener().apply { listener() }

/**
 *
 */
class PickerListener : MediaPickerListener {

    private var mNoAccess: () -> Unit = {}
    private var mOnFileReceivedFailed: () -> Unit = {}
    private var mOnFileReceived: (files: List<Uri?>) -> Unit = {}

    override fun onFileReceived(files: List<Uri?>) {
        mOnFileReceived.invoke(files)
    }

    override fun noAccess() {
        mNoAccess.invoke()
    }

    override fun onFileReceivedFailed() {
        mOnFileReceivedFailed.invoke()
    }

    fun onFileReceived(func: (files: List<Uri?>) -> Unit) {
        mOnFileReceived = func
    }

    fun noAccess(func: () -> Unit) {
        mNoAccess = func
    }

    fun onFileReceivedFailed(func: () -> Unit) {
        mOnFileReceivedFailed = func
    }
}

/**
 * To support Java
 */
object MediaPicker {

    @JvmStatic
    fun getImage(
        activity: FragmentActivity,
        listener: MediaPickerListener
    ) {
        activity.getMedia(listener)
    }
}