package org.qas.qtest.api.services.requirement;

import org.qas.api.AuthClientException;
import org.qas.api.AuthServiceException;
import org.qas.api.ClientConfiguration;
import org.qas.api.Request;
import org.qas.api.internal.util.google.base.Throwables;
import org.qas.qtest.api.auth.DefaultQTestCredentialsProviderChain;
import org.qas.qtest.api.auth.QTestCredentials;
import org.qas.qtest.api.auth.QTestCredentialsProvider;
import org.qas.qtest.api.auth.StaticQTestCredentialsProvider;
import org.qas.qtest.api.internal.QTestApiWebServiceClient;
import org.qas.qtest.api.internal.model.CreateObjectCommentRequest;
import org.qas.qtest.api.internal.model.Field;
import org.qas.qtest.api.internal.model.transform.CreateObjectCommentRequestMarshaller;
import org.qas.qtest.api.internal.model.transform.ListFieldJsonUnmarshaller;
import org.qas.qtest.api.services.requirement.model.CreateRequirementRequest;
import org.qas.qtest.api.services.requirement.model.GetRequirementFieldsRequest;
import org.qas.qtest.api.services.requirement.model.Requirement;
import org.qas.qtest.api.services.requirement.model.transform.CreateRequirementRequestMarshaller;
import org.qas.qtest.api.services.requirement.model.transform.GetRequirementFieldsRequestMarshaller;
import org.qas.qtest.api.services.requirement.model.transform.RequirementJsonUnmarshaller;

import java.util.List;

/**
 * RequirementServiceClient
 *
 * @author Dzung Nguyen
 * @version $Id RequirementServiceClient 2014-05-19 11:36:30z dungvnguyen $
 * @since 1.0
 */
public class RequirementServiceClient extends QTestApiWebServiceClient<RequirementServiceClient>
  implements RequirementService {
  /**
   * Constructs a new client to invoke service method on RequirementService using
   * the default qTest credentials provider and default client configuration options.
   */
  public RequirementServiceClient() {
    this(new DefaultQTestCredentialsProviderChain(), new ClientConfiguration());
  }

  /**
   * Constructs a new client to invoke service method on RequirementService using
   * the default qTest credentials provider and client configuration options.
   *
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to RequirementService
   */
  public RequirementServiceClient(ClientConfiguration clientConfiguration) {
    this(new DefaultQTestCredentialsProviderChain(), clientConfiguration);
  }

  /**
   * Constructs a new client to invoke service method on RequirementService using
   * the specified qTest credentials.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   */
  public RequirementServiceClient(QTestCredentials credentials) {
    this(credentials, new ClientConfiguration());
  }

  /**
   * Constructs a new client to invoke service method on RequirementService using
   * the specified qTest credentials and client configuration options.
   *
   * @param credentials         The qTest credentials which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to RequirementService
   */
  public RequirementServiceClient(QTestCredentials credentials, ClientConfiguration clientConfiguration) {
    super(clientConfiguration);
    this.credentialsProvider = new StaticQTestCredentialsProvider(credentials);
    init();
  }

  /**
   * Constructs a new client to invoke service method on RequirementService using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   */
  public RequirementServiceClient(QTestCredentialsProvider credentialsProvider) {
    this(credentialsProvider, new ClientConfiguration());
  }

  /**
   * Constructs a new client to invoke service method on RequirementService using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to RequirementService
   */
  public RequirementServiceClient(QTestCredentialsProvider credentialsProvider, ClientConfiguration clientConfiguration) {
    super(clientConfiguration);
    this.credentialsProvider = credentialsProvider;

    init();
  }

  @Override
  public List<Field> getRequirementFields(GetRequirementFieldsRequest getRequirementFieldsRequest) throws AuthServiceException {
    try {
      Request request = new GetRequirementFieldsRequestMarshaller().marshall(getRequirementFieldsRequest);
      return invoke(request, ListFieldJsonUnmarshaller.getInstance());
    } catch (Exception ex) {
      Throwables.propagateIfInstanceOf(ex, AuthClientException.class);
      throw new AuthClientException("Unknown exception occurs during getting requirement fields", ex);
    }
  }

  @Override
  public Requirement createRequirement(CreateRequirementRequest createRequirementRequest) throws AuthServiceException {
    try {
      Request request = new CreateRequirementRequestMarshaller().marshall(createRequirementRequest);
      return invoke(request, RequirementJsonUnmarshaller.getInstance());
    } catch (Exception ex) {
      Throwables.propagateIfInstanceOf(ex, AuthClientException.class);
      throw new AuthClientException("Unknown exception occurs during create requirement data", ex);
    }
  }

  @Override
  public Requirement addComment(CreateObjectCommentRequest createObjectCommentRequest) throws AuthServiceException {
    try {
      Request request = new CreateObjectCommentRequestMarshaller("RequirementService").marshall(createObjectCommentRequest);
      return invoke(request, RequirementJsonUnmarshaller.getInstance());
    } catch (Exception ex) {
      Throwables.propagateIfInstanceOf(ex, AuthClientException.class);
      throw new AuthClientException("Unknown exception occurs dủing creating requirement comment", ex);
    }
  }
}
