package org.qas.qtest.api.services.requirement;

import org.qas.api.AuthServiceException;
import org.qas.api.ClientConfiguration;
import org.qas.api.handler.AsyncHandler;
import org.qas.qtest.api.auth.DefaultQTestCredentialsProviderChain;
import org.qas.qtest.api.auth.QTestCredentials;
import org.qas.qtest.api.auth.QTestCredentialsProvider;
import org.qas.qtest.api.auth.StaticQTestCredentialsProvider;
import org.qas.qtest.api.internal.model.CreateObjectCommentRequest;
import org.qas.qtest.api.internal.model.Field;
import org.qas.qtest.api.services.requirement.model.CreateRequirementRequest;
import org.qas.qtest.api.services.requirement.model.GetRequirementFieldsRequest;
import org.qas.qtest.api.services.requirement.model.Requirement;

import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

/**
 * RequirementServiceAsyncClient
 *
 * @author Dzung Nguyen
 * @version $Id RequirementServiceAsyncClient 2014-05-19 12:24:30z dungvnguyen $
 * @since 1.0
 */
public class RequirementServiceAsyncClient extends RequirementServiceClient
    implements RequirementServiceAsync {
  /**
   * Constructs a new client to invoke service method on RequirementService using
   * the default qTest credentials provider and default client configuration options.
   */
  public RequirementServiceAsyncClient() {
    this(new DefaultQTestCredentialsProviderChain(), new ClientConfiguration(), Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on RequirementService using
   * the default qTest credentials provider and default client configuration options.
   *
   * @param executorService the executor service for executing asynchronous request.
   */
  public RequirementServiceAsyncClient(ExecutorService executorService) {
    this(new DefaultQTestCredentialsProviderChain(), new ClientConfiguration(), executorService);
  }

  /**
   * Constructs a new client to invoke service method on RequirementService using
   * the default qTest credentials provider and client configuration options.
   *
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to RequirementService
   */
  public RequirementServiceAsyncClient(ClientConfiguration clientConfiguration) {
    this(new DefaultQTestCredentialsProviderChain(), clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on RequirementService using
   * the default qTest credentials provider and client configuration options.
   *
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to RequirementService.
   * @param executorService the executor service for executing asynchronous request.
   */
  public RequirementServiceAsyncClient(ClientConfiguration clientConfiguration, ExecutorService executorService) {
    this(new DefaultQTestCredentialsProviderChain(), clientConfiguration, executorService);
  }

  /**
   * Constructs a new client to invoke service method on RequirementService using
   * the specified qTest credentials.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   */
  public RequirementServiceAsyncClient(QTestCredentials credentials) {
    this(credentials, new ClientConfiguration(), Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on RequirementService using
   * the specified qTest credentials.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param executorService the executor service for executing asynchronous request.
   */
  public RequirementServiceAsyncClient(QTestCredentials credentials, ExecutorService executorService) {
    this(credentials, new ClientConfiguration(), executorService);
  }

  /**
   * Constructs a new client to invoke service method on RequirementService using
   * the specified qTest credentials and client configuration options.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to RequirementService
   */
  public RequirementServiceAsyncClient(QTestCredentials credentials, ClientConfiguration clientConfiguration) {
    this(credentials, clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on RequirementService using
   * the specified qTest credentials and client configuration options.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to RequirementService
   * @param executorService the executor service for executing asynchronous request.
   */
  public RequirementServiceAsyncClient(QTestCredentials credentials, ClientConfiguration clientConfiguration,
                                   ExecutorService executorService) {
    this(new StaticQTestCredentialsProvider(credentials), clientConfiguration, executorService);
  }

  /**
   * Constructs a new client to invoke service method on RequirementService using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to RequirementService
   */
  public RequirementServiceAsyncClient(QTestCredentialsProvider credentialsProvider,
                                   ClientConfiguration clientConfiguration) {
    this(credentialsProvider, clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on RequirementService using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to RequirementService
   * @param executorService the executor service for executing asynchronous request.
   */
  public RequirementServiceAsyncClient(QTestCredentialsProvider credentialsProvider,
                                   ClientConfiguration clientConfiguration, ExecutorService executorService) {
    super(credentialsProvider, clientConfiguration);
    this.executorService = executorService;
  }

  //~ implements methods ======================================================
  @Override
  public Future<List<Field>> getRequirementFieldsAsync(final GetRequirementFieldsRequest getRequirementFieldsRequest)
    throws AuthServiceException {
    return executorService.submit(new Callable<List<Field>>() {
      @Override
      public List<Field> call() throws Exception {
        return getRequirementFields(getRequirementFieldsRequest);
      }
    });
  }

  @Override
  public Future<List<Field>> getRequirementFieldsAsync(final GetRequirementFieldsRequest getRequirementFieldsRequest,
                                                       final AsyncHandler<GetRequirementFieldsRequest, List<Field>> asyncHandler)
      throws AuthServiceException {
    return executorService.submit(new Callable<List<Field>>() {
      @Override
      public List<Field> call() throws Exception {
        List<Field> result;

        try {
          result = getRequirementFields(getRequirementFieldsRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(getRequirementFieldsRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<Requirement> createRequirementAsync(final CreateRequirementRequest createRequirementRequest)
      throws AuthServiceException {
    return executorService.submit(new Callable<Requirement>() {
      @Override
      public Requirement call() throws Exception {
        return createRequirement(createRequirementRequest);
      }
    });
  }

  @Override
  public Future<Requirement> createRequirementAsync(final CreateRequirementRequest createRequirementRequest,
                                                    final AsyncHandler<CreateRequirementRequest, Requirement> asyncHandler)
      throws AuthServiceException {
    return executorService.submit(new Callable<Requirement>() {
      @Override
      public Requirement call() throws Exception {
        Requirement result;

        try {
          result = createRequirement(createRequirementRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(createRequirementRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<Requirement> addCommentAsync(final CreateObjectCommentRequest createObjectCommentRequest) throws AuthServiceException {
    return executorService.submit(new Callable<Requirement>() {
      @Override
      public Requirement call() throws Exception {
        return addComment(createObjectCommentRequest);
      }
    });
  }

  @Override
  public Future<Requirement> addCommentAsync(final CreateObjectCommentRequest createObjectCommentRequest,
                                             final AsyncHandler<CreateObjectCommentRequest, Requirement> asyncHandler) throws AuthServiceException {
    return executorService.submit(new Callable<Requirement>() {
      @Override
      public Requirement call() throws Exception {
        Requirement result;

        try {
          result = addComment(createObjectCommentRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(createObjectCommentRequest, result);
        return result;
      }
    });
  }
}
