package org.qas.qtest.api.services.project;

import org.qas.api.AuthServiceException;
import org.qas.api.ClientConfiguration;
import org.qas.api.handler.AsyncHandler;
import org.qas.qtest.api.auth.DefaultQTestCredentialsProviderChain;
import org.qas.qtest.api.auth.QTestCredentials;
import org.qas.qtest.api.auth.QTestCredentialsProvider;
import org.qas.qtest.api.auth.StaticQTestCredentialsProvider;
import org.qas.qtest.api.internal.model.Field;
import org.qas.qtest.api.services.project.model.*;

import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

/**
 * ProjectServiceAsyncClient
 *
 * @author Dzung Nguyen
 * @version $Id ProjectServiceAsyncClient 2014-03-27 22:28:30z dungvnguyen $
 * @since 1.0
 */
public class ProjectServiceAsyncClient extends ProjectServiceClient implements ProjectServiceAsync {
  /**
   * Constructs a new client to invoke service method on ProjectService using
   * the default qTest credentials provider and default client configuration options.
   */
  public ProjectServiceAsyncClient() {
    this(new DefaultQTestCredentialsProviderChain(), new ClientConfiguration(), Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on ProjectService using
   * the default qTest credentials provider and default client configuration options.
   *
   * @param executorService the executor service for executing asynchronous request.
   */
  public ProjectServiceAsyncClient(ExecutorService executorService) {
    this(new DefaultQTestCredentialsProviderChain(), new ClientConfiguration(), executorService);
  }

  /**
   * Constructs a new client to invoke service method on ProjectService using
   * the default qTest credentials provider and client configuration options.
   *
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to ProjectService
   */
  public ProjectServiceAsyncClient(ClientConfiguration clientConfiguration) {
    this(new DefaultQTestCredentialsProviderChain(), clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on ProjectService using
   * the default qTest credentials provider and client configuration options.
   *
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to ProjectService.
   * @param executorService the executor service for executing asynchronous request.
   */
  public ProjectServiceAsyncClient(ClientConfiguration clientConfiguration, ExecutorService executorService) {
    this(new DefaultQTestCredentialsProviderChain(), clientConfiguration, executorService);
  }

  /**
   * Constructs a new client to invoke service method on ProjectService using
   * the specified qTest credentials.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   */
  public ProjectServiceAsyncClient(QTestCredentials credentials) {
    this(credentials, new ClientConfiguration(), Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on ProjectService using
   * the specified qTest credentials.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param executorService the executor service for executing asynchronous request.
   */
  public ProjectServiceAsyncClient(QTestCredentials credentials, ExecutorService executorService) {
    this(credentials, new ClientConfiguration(), executorService);
  }

  /**
   * Constructs a new client to invoke service method on ProjectService using
   * the specified qTest credentials and client configuration options.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to ProjectService
   */
  public ProjectServiceAsyncClient(QTestCredentials credentials, ClientConfiguration clientConfiguration) {
    this(credentials, clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on ProjectService using
   * the specified qTest credentials and client configuration options.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to ProjectService
   * @param executorService the executor service for executing asynchronous request.
   */
  public ProjectServiceAsyncClient(QTestCredentials credentials, ClientConfiguration clientConfiguration,
                                   ExecutorService executorService) {
    this(new StaticQTestCredentialsProvider(credentials), clientConfiguration, executorService);
  }

  /**
   * Constructs a new client to invoke service method on ProjectServiceAsync using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to ProjectService
   */
  public ProjectServiceAsyncClient(QTestCredentialsProvider credentialsProvider,
                                   ClientConfiguration clientConfiguration) {
    this(credentialsProvider, clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on ProjectServiceAsync using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to ProjectService
   * @param executorService the executor service for executing asynchronous request.
   */
  public ProjectServiceAsyncClient(QTestCredentialsProvider credentialsProvider,
                                   ClientConfiguration clientConfiguration, ExecutorService executorService) {
    super(credentialsProvider, clientConfiguration);
    this.executorService = executorService;
  }

  @Override
  public Future<List<Project>> listProjectAsync(final ListProjectRequest listProjectRequest) throws AuthServiceException {
    return executorService.submit(new Callable<List<Project>>() {
      @Override
      public List<Project> call() throws Exception {
        return listProject(listProjectRequest);
      }
    });
  }

  @Override
  public Future<List<Project>> listProjectAsync(final ListProjectRequest listProjectRequest,
                                                final AsyncHandler<ListProjectRequest, List<Project>> asyncHandler) throws AuthServiceException {
    return executorService.submit(new Callable<List<Project>>() {
      @Override
      public List<Project> call() throws Exception {
        List<Project> result;

        try {
          result = listProject(listProjectRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(listProjectRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<Project> createProjectAsync(final CreateProjectRequest createProjectRequest)
    throws AuthServiceException {
    return executorService.submit(new Callable<Project>() {
      @Override
      public Project call() throws Exception {
        return createProject(createProjectRequest);
      }
    });
  }

  @Override
  public Future<Project> createProjectAsync(final CreateProjectRequest createProjectRequest,
                                            final AsyncHandler<CreateProjectRequest, Project> asyncHandler)
    throws AuthServiceException {
    return executorService.submit(new Callable<Project>() {
      @Override
      public Project call() throws Exception {
        Project result;

        try {
          result = createProject(createProjectRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(createProjectRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<Module> createModuleAsync(final CreateModuleRequest createModuleRequest)
    throws AuthServiceException {
    return executorService.submit(new Callable<Module>(){
      @Override
      public Module call() throws Exception {
        return createModule(createModuleRequest);
      }
    });
  }

  @Override
  public Future<Module> createModuleAsync(final CreateModuleRequest createModuleRequest,
                                          final AsyncHandler<CreateModuleRequest, Module> asyncHandler)
    throws AuthServiceException {
    return executorService.submit(new Callable<Module>() {
      @Override
      public Module call() throws Exception {
        Module result;

        try {
          result = createModule(createModuleRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(createModuleRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<Module> getModuleAsync(final GetModuleRequest getModuleRequest) throws AuthServiceException {
    return executorService.submit(new Callable<Module>() {
      @Override
      public Module call() throws Exception {
        return getModule(getModuleRequest);
      }
    });
  }

  @Override
  public Future<Module> getModuleAsync(final GetModuleRequest getModuleRequest,
                                       final AsyncHandler<GetModuleRequest, Module> asyncHandler)
      throws AuthServiceException {
    return executorService.submit(new Callable<Module>() {
      @Override
      public Module call() throws Exception {
        Module result;

        try {
          result = getModule(getModuleRequest);
        } catch (Exception e) {
          asyncHandler.onError(e);
          throw e;
        }

        asyncHandler.onSuccess(getModuleRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<List<Module>> listModuleAsync(final ListModuleRequest listModuleRequest)
      throws AuthServiceException {
    return executorService.submit(new Callable<List<Module>>() {
      @Override
      public List<Module> call() throws Exception {
        return listModule(listModuleRequest);
      }
    });
  }

  @Override
  public Future<List<Module>> listModuleAsync(final ListModuleRequest listModuleRequest,
                                              final AsyncHandler<ListModuleRequest, List<Module>> asyncHandler)
      throws AuthServiceException {
    return executorService.submit(new Callable<List<Module>>() {
      @Override
      public List<Module> call() throws Exception {
        List<Module> result;

        try {
          result = listModule(listModuleRequest);
        } catch (Exception e) {
          asyncHandler.onError(e);
          throw e;
        }

        asyncHandler.onSuccess(listModuleRequest, result);
        return result;
      }
    });
  }
  
  @Override
  public Future<Field> createCustomFieldAsync(final CreateCustomFieldRequest createCustomFieldRequest) throws AuthServiceException {
    return executorService.submit(new Callable<Field>() {
      @Override
      public Field call() throws Exception {
        return createCustomField(createCustomFieldRequest);
      }
    });
  }

  @Override
  public Future<Field> createCustomFieldAsync(final CreateCustomFieldRequest createCustomFieldRequest,
                                              final AsyncHandler<CreateCustomFieldRequest, Field> asyncHandler)
      throws AuthServiceException {
    return executorService.submit(new Callable<Field>() {
      @Override
      public Field call() throws Exception {
        final Field result;

        try {
          result = createCustomField(createCustomFieldRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(createCustomFieldRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<Field> updateSystemFieldAsync(final UpdateSystemFieldRequest updateSystemFieldRequest) throws AuthServiceException {
    return executorService.submit(new Callable<Field>() {
      @Override
      public Field call() throws Exception {
        return updateSystemField(updateSystemFieldRequest);
      }
    });
  }

  @Override
  public Future<Field> updateSystemFieldAsync(final UpdateSystemFieldRequest updateSystemFieldRequest,
                                              final AsyncHandler<UpdateSystemFieldRequest, Field> asyncHandler)
      throws AuthServiceException {
    return executorService.submit(new Callable<Field>() {
      @Override
      public Field call() throws Exception {
        final Field result;

        try {
          result = updateSystemField(updateSystemFieldRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(updateSystemFieldRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<List<Field>> getFieldsAsync(final GetFieldsRequest getFieldsRequest) throws AuthServiceException {
    return executorService.submit(new Callable<List<Field>>() {
      @Override
      public List<Field> call() throws Exception {
        return getFields(getFieldsRequest);
      }
    });
  }

  @Override
  public Future<List<Field>> getFieldsAsync(final GetFieldsRequest getFieldsRequest,
                                            final AsyncHandler<GetFieldsRequest, List<Field>> asyncHandler)
      throws AuthServiceException {
    return executorService.submit(new Callable<List<Field>>() {
      @Override
      public List<Field> call() throws Exception {
        final List<Field> result;

        try {
          result = getFields(getFieldsRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(getFieldsRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<UserPermissions> getUserPermissionsAsync(final GetUserPermissionsRequest getUserPermissionsRequest) throws AuthServiceException {
    return executorService.submit(new Callable<UserPermissions>() {
      @Override
      public UserPermissions call() throws Exception {
        return getUserPermissions(getUserPermissionsRequest);
      }
    });
  }

  @Override
  public Future<UserPermissions> getUserPermissionsAsync(final GetUserPermissionsRequest getUserPermissionsRequest,
                                                         final AsyncHandler<GetUserPermissionsRequest, UserPermissions> asyncHandler)
      throws AuthServiceException {
    return executorService.submit(new Callable<UserPermissions>() {
      @Override
      public UserPermissions call() throws Exception {
        final UserPermissions result;

        try {
          result = getUserPermissions(getUserPermissionsRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(getUserPermissionsRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<List<UserPermissions>> listUserPermissionsAsync(final ListUserPermissionsRequest listUserPermissionsRequest)
      throws AuthServiceException {
    return executorService.submit(new Callable<List<UserPermissions>>() {
      @Override
      public List<UserPermissions> call() throws Exception {
        return listUserPermissions(listUserPermissionsRequest);
      }
    });
  }

  @Override
  public Future<List<UserPermissions>> listUserPermissionsAsync(final ListUserPermissionsRequest listUserPermissionsRequest,
                                                                final AsyncHandler<ListUserPermissionsRequest, List<UserPermissions>> asyncHandler)
      throws AuthServiceException {
    return executorService.submit(new Callable<List<UserPermissions>>() {
      @Override
      public List<UserPermissions> call() throws Exception {
        final List<UserPermissions> result;

        try {
          result = listUserPermissions(listUserPermissionsRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(listUserPermissionsRequest, result);
        return result;
      }
    });
  }
}
