package org.qas.qtest.api.services.plan;

import org.qas.api.AuthServiceException;
import org.qas.api.ClientConfiguration;
import org.qas.api.handler.AsyncHandler;
import org.qas.qtest.api.auth.DefaultQTestCredentialsProviderChain;
import org.qas.qtest.api.auth.QTestCredentials;
import org.qas.qtest.api.auth.QTestCredentialsProvider;
import org.qas.qtest.api.auth.StaticQTestCredentialsProvider;
import org.qas.qtest.api.services.plan.model.Build;
import org.qas.qtest.api.services.plan.model.CreateBuildRequest;
import org.qas.qtest.api.services.plan.model.CreateReleaseRequest;
import org.qas.qtest.api.services.plan.model.Release;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

/**
 * TestPlanServiceAsyncClient
 *
 * @author Dzung Nguyen
 * @version $Id TestPlanServiceAsyncClient 2014-07-18 15:19:30z dungvnguyen $
 * @since 1.0
 */
public class TestPlanServiceAsyncClient extends TestPlanServiceClient
  implements TestPlanServiceAsync {
  /**
   * Constructs a new client to invoke service method on TestPlanService using
   * the default qTest credentials provider and default client configuration options.
   */
  public TestPlanServiceAsyncClient() {
    this(new DefaultQTestCredentialsProviderChain(), new ClientConfiguration(), Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on TestPlanService using
   * the default qTest credentials provider and default client configuration options.
   *
   * @param executorService the executor service for executing asynchronous request.
   */
  public TestPlanServiceAsyncClient(ExecutorService executorService) {
    this(new DefaultQTestCredentialsProviderChain(), new ClientConfiguration(), executorService);
  }

  /**
   * Constructs a new client to invoke service method on TestPlanService using
   * the default qTest credentials provider and client configuration options.
   *
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to TestPlanService
   */
  public TestPlanServiceAsyncClient(ClientConfiguration clientConfiguration) {
    this(new DefaultQTestCredentialsProviderChain(), clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on TestPlanService using
   * the default qTest credentials provider and client configuration options.
   *
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to TestPlanService.
   * @param executorService the executor service for executing asynchronous request.
   */
  public TestPlanServiceAsyncClient(ClientConfiguration clientConfiguration, ExecutorService executorService) {
    this(new DefaultQTestCredentialsProviderChain(), clientConfiguration, executorService);
  }

  /**
   * Constructs a new client to invoke service method on TestPlanService using
   * the specified qTest credentials.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   */
  public TestPlanServiceAsyncClient(QTestCredentials credentials) {
    this(credentials, new ClientConfiguration(), Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on TestPlanService using
   * the specified qTest credentials.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param executorService the executor service for executing asynchronous request.
   */
  public TestPlanServiceAsyncClient(QTestCredentials credentials, ExecutorService executorService) {
    this(credentials, new ClientConfiguration(), executorService);
  }

  /**
   * Constructs a new client to invoke service method on TestPlanService using
   * the specified qTest credentials and client configuration options.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to TestPlanService
   */
  public TestPlanServiceAsyncClient(QTestCredentials credentials, ClientConfiguration clientConfiguration) {
    this(credentials, clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on TestPlanService using
   * the specified qTest credentials and client configuration options.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to TestPlanService
   * @param executorService the executor service for executing asynchronous request.
   */
  public TestPlanServiceAsyncClient(QTestCredentials credentials, ClientConfiguration clientConfiguration,
                                  ExecutorService executorService) {
    this(new StaticQTestCredentialsProvider(credentials), clientConfiguration, executorService);
  }

  /**
   * Constructs a new client to invoke service method on TestPlanService using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to TestPlanService
   */
  public TestPlanServiceAsyncClient(QTestCredentialsProvider credentialsProvider,
                                  ClientConfiguration clientConfiguration) {
    this(credentialsProvider, clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on TestPlanService using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to TestPlanService
   * @param executorService the executor service for executing asynchronous request.
   */
  public TestPlanServiceAsyncClient(QTestCredentialsProvider credentialsProvider,
                                  ClientConfiguration clientConfiguration, ExecutorService executorService) {
    super(credentialsProvider, clientConfiguration);
    this.executorService = executorService;
  }

  @Override
  public Future<Build> createBuildAsync(final CreateBuildRequest createBuildRequest)
      throws AuthServiceException {
    return executorService.submit(new Callable<Build>() {
      @Override
      public Build call() throws Exception {
        return createBuild(createBuildRequest);
      }
    });
  }

  @Override
  public Future<Build> createBuildAsync(final CreateBuildRequest createBuildRequest,
                                        final AsyncHandler<CreateBuildRequest, Build> asyncHandler)
      throws AuthServiceException {
    return executorService.submit(new Callable<Build>() {
      @Override
      public Build call() throws Exception {
        final Build result;

        try {
          result = createBuild(createBuildRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(createBuildRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<Release> createReleaseAsync(final CreateReleaseRequest createReleaseRequest)
      throws AuthServiceException {
    return executorService.submit(new Callable<Release>() {
      @Override
      public Release call() throws Exception {
        return createRelease(createReleaseRequest);
      }
    });
  }

  @Override
  public Future<Release> createReleaseAsync(final CreateReleaseRequest createReleaseRequest,
                                            final AsyncHandler<CreateReleaseRequest, Release> asyncHandler)
      throws AuthServiceException {
    return executorService.submit(new Callable<Release>() {
      @Override
      public Release call() throws Exception {
        final Release result;

        try {
          result = createRelease(createReleaseRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(createReleaseRequest, result);
        return result;
      }
    });
  }
}
