package org.qas.qtest.api.auth;

import org.qas.api.AuthClientException;

import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * QTestCredentialsProviderChain
 *
 * @author Dzung Nguyen
 * @version $Id QTestCredentialsProviderChain 2014-03-27 13:43:30z dungvnguyen $
 * @since 1.0
 */
public class QTestCredentialsProviderChain implements QTestCredentialsProvider {
  //~ class properties ========================================================
  private static final Logger LOG = Logger.getLogger(QTestCredentialsProviderChain.class.getName());

  private List<QTestCredentialsProvider> credentialsProviders = new LinkedList<QTestCredentialsProvider>();
  //~ class members ===========================================================
  /**
   * Constructs a new QTestCredentialsProviderChain with the specified
   * credential providers. When credentials are requested from this provider,
   * it will call each of these credential providers in the same order
   * specified here until one of them returns qTest security credentials.
   *
   * @param credentialsProviders
   *            The chain of credentials providers.
   */
  public QTestCredentialsProviderChain(QTestCredentialsProvider...credentialsProviders) {
    if (credentialsProviders == null || credentialsProviders.length == 0) {
      throw new IllegalArgumentException("No credentials providers specified.");
    }

    for (QTestCredentialsProvider credentialsProvider : credentialsProviders) {
      this.credentialsProviders.add(credentialsProvider);
    }
  }

  public QTestCredentials getCredentials() {
    for (QTestCredentialsProvider credentialsProvider : credentialsProviders) {
      try {
        QTestCredentials credentials = credentialsProvider.getCredentials();
        if (credentials.getToken() != null) {
          if (LOG.isLoggable(Level.FINE)) {
            LOG.fine("Loading credentials from " + credentialsProvider.toString());
          }

          return credentials;
        }
      } catch (Exception ex) {
        if (LOG.isLoggable(Level.FINE)) {
          LOG.fine("Unable to load credentials from " + credentialsProvider.toString() + ":" + ex.getMessage());
        }
      }
    }

    throw new AuthClientException("Unable to load qTest credentials from any provider in the chain.");
  }

  public void refresh() {
    for (QTestCredentialsProvider credentialsProvider : credentialsProviders) {
      credentialsProvider.refresh();
    }
  }
}
