package org.qas.api.support;

import org.qas.api.ApiServiceRequest;
import org.qas.api.Request;
import org.qas.api.http.HttpMethod;
import org.qas.api.internal.util.google.base.Preconditions;
import org.qas.api.internal.util.google.base.Strings;

import java.io.InputStream;
import java.net.URI;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * RequestSupport
 *
 * @author Dzung Nguyen
 * @version $Id RequestSupport 2014-03-27 10:45:30z dungvnguyen $
 * @since 1.0
 */
public class RequestSupport implements Request {
  //~ class properties ========================================================
  private final String serviceName;
  private final ApiServiceRequest originalRequest;
  private String resourcePath;
  private HttpMethod httpMethod = HttpMethod.POST;
  private URI endpoint;
  private int timeOffset;
  private InputStream content;
  private Map<String, String> parameters = new HashMap<String, String>();
  private Map<String, String> headers = new HashMap<String, String>();

  //~ class members ===========================================================
  /**
   * Creates the {@link RequestSupport} object from the given service name.
   *
   * @param serviceName the given service name.
   */
  public RequestSupport(ApiServiceRequest originalRequest, String serviceName) {
    Preconditions.checkNotNull(serviceName, "The service name must not null.");

    // set the service name.
    this.serviceName = serviceName;
    this.originalRequest = originalRequest;
  }

  @Override
  public String getServiceName() {
    return serviceName;
  }

  @Override
  public String getResourcePath() {
    return resourcePath;
  }

  @Override
  public void setResourcePath(String resourcePath) {
    this.resourcePath = resourcePath;
  }

  @Override
  public Request withResourcePath(String resourcePath) {
    setResourcePath(resourcePath);
    return this;
  }

  @Override
  public HttpMethod getHttpMethod() {
    return httpMethod;
  }

  @Override
  public void setHttpMethod(HttpMethod httpMethod) {
    Preconditions.checkNotNull(httpMethod, "The HTTP Method must not be null.");
    this.httpMethod = httpMethod;
  }

  @Override
  public Request withHttpMethod(HttpMethod httpMethod) {
    setHttpMethod(httpMethod);
    return this;
  }

  @Override
  public void setEndpoint(URI endpoint) {
    Preconditions.checkNotNull(endpoint);
    this.endpoint = endpoint;
  }

  @Override
  public URI getEndpoint() {
    return endpoint;
  }

  @Override
  public Request withEndpoint(URI endpoint) {
    setEndpoint(endpoint);
    return this;
  }

  @Override
  public void setTimeOffset(int timeOffset) {
    this.timeOffset = timeOffset;
  }

  @Override
  public int getTimeOffset() {
    return timeOffset;
  }

  @Override
  public Request withTimeOffset(int timeOffset) {
    setTimeOffset(timeOffset);
    return this;
  }

  @Override
  public void setContent(InputStream content) {
    this.content = content;
  }

  @Override
  public InputStream getContent() {
    return content;
  }

  @Override
  public Request withContent(InputStream content) {
    setContent(content);
    return this;
  }

  @Override
  public void setHeader(String name, String value) {
    headers.put(name, value);
  }

  @Override
  public void setHeaders(Map<String, String> headers) {
    this.headers.clear();

    for (Map.Entry<String, String> entry : headers.entrySet()) {
      this.headers.put(entry.getKey(), entry.getValue());
    }
  }

  @Override
  public Request withHeader(String name, String value) {
    setHeader(name, value);
    return this;
  }

  @Override
  public Request withHeaders(Map<String, String> headers) {
    setHeaders(headers);
    return this;
  }

  @Override
  public Map<String, String> getHeaders() {
    return Collections.unmodifiableMap(headers);
  }

  @Override
  public void addParameter(String name, String value) {
    parameters.put(name, value);
  }

  @Override
  public Request withParameter(String name, String value) {
    addParameter(name, value);
    return this;
  }

  @Override
  public void setParameters(Map<String, String> parameters) {
    this.parameters.clear();

    for (Map.Entry<String, String> entry: parameters.entrySet()) {
      this.parameters.put(entry.getKey(), entry.getValue());
    }
  }

  @Override
  public Request withParameters(Map<String, String> parameters) {
    setParameters(parameters);
    return this;
  }

  @Override
  public Map<String, String> getParameters() {
    return Collections.unmodifiableMap(this.parameters);
  }

  @Override
  public ApiServiceRequest getOriginalRequest() {
    return originalRequest;
  }

  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder("<request>\n");

    // build request object.
    builder.append("\t<service-name value=\"").append(serviceName).append("\"/>\n");
    builder.append("\t<http-method value=\"").append(httpMethod.name()).append("\"/>\n");
    builder.append("\t<endpoint value=\"").append(endpoint.toString()).append("\"/>\n");

    // append resource path.
    if (!Strings.isNullOrEmpty(resourcePath)) {
      builder.append("\t<resource-path value=\"").append(resourcePath).append("\"/>\n");
    }

    builder.append("\t<time-offset value=\"").append(timeOffset).append("\"/>\n");
    builder.append("\t<headers>").append(headers.toString()).append("</headers>\n");
    builder.append("\t<parameters>").append(parameters.toString()).append("</parameters>\n");

    builder.append("</request>");

    return builder.toString();
  }
}
