package org.qas.api.net;

import org.qas.api.http.FastUrlEncoder;
import org.qas.api.internal.util.google.base.Charsets;
import org.qas.api.internal.util.google.base.Preconditions;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.nio.charset.Charset;

/**
 * UriEncoder
 *
 * @author Dzung Nguyen
 * @version $Id UriEncoder 2014-03-27 09:35:30z dungvnguyen $
 * @since 1.0
 */
public final class UriEncoder {
  //~ class properties ========================================================
  /**
   * The default character encoding.
   */
  public static final Charset DEFAULT_ENCODING = Charsets.UTF_8;

  //~ class members ===========================================================
  private UriEncoder() {
    throw new AssertionError("UriEncoder utility class must not be instantiated.");
  }

  /**
   * Percent-encodes a Unicode string into a US-ASCII string.
   *
   * <p>This replaces '&nbsp;' with '%20'. So this method should be used for
   * non application/x-www-form-urlencoded strings such as host and path.</p>
   *
   * @param source a Unicode string.
   * @return a percent-encoded US-ASCII string.
   */
  public static String encode(String source) {
    return encode(source, DEFAULT_ENCODING);
  }

  /**
   * Percent-encodes a Unicode string into US-ASCII string. The specified encoding
   * is used to determine how non-US-ASCII and reserved characters should be
   * represented as consecutive sequences of the form "%<i>XX</i>"
   *
   * <p>This replaces '&nbsp;' with '%20'. So this method should be used for
   * non application/x-www-form-urlencoded strings such as host and path.</p>
   *
   * @param source a String to encode.
   * @param encoding a character encoding.
   * @return a percent-encoded US-ASCII string.
   */
  public static String encode(String source, Charset encoding) {
    Preconditions.checkNotNull(source);
    Preconditions.checkNotNull(encoding);

    try {
      return FastUrlEncoder.uriEncode(source, encoding.name());
    } catch (UnsupportedEncodingException ueex) {
      throw new AssertionError(ueex);
    }
  }

  /**
   * Percent-decodes a US-ASCII string into a Unicode string.
   *
   * <p>This replaces '%20' with '&nbsp;'. So this method should be used
   * for non application/x-www-form-urlencoded string such as host and path.</p>
   *
   * @param source a percent-encoded US-ASCII string.
   * @return a Unicode string.
   */
  public static String decode(String source) {
    return decode(source, DEFAULT_ENCODING);
  }

  /**
   * Percent-decodes a US-ASCII string into Unicode string. The specified encoding
   * is used to determine what character are represented by any consecutive
   * sequences of the form "%<i>XX</i>".
   *
   * <p>This replaces '%20' with '&nbsp;'. So this method should be used
   * for non application/x-www-form-urlencoded string such as host and path.</p>
   *
   * @param source a percent-encoded US-ASCII string.
   * @param encoding a character encoding.
   * @return a unicode string.
   */
  public static String decode(String source, Charset encoding) {
    Preconditions.checkNotNull(source);
    Preconditions.checkNotNull(encoding);

    try {
      return URLDecoder.decode(source, encoding.name());
    } catch (UnsupportedEncodingException uuex) {
      throw new AssertionError(uuex);
    }
  }
}
