package org.qas.api.internal.util.key.support;

import org.qas.api.internal.util.Encoders;
import org.qas.api.internal.util.key.UniqueKey;
import org.qas.api.internal.util.key.UniqueKeyImportException;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;

/**
 * MachineKey
 *
 * @author Dzung Nguyen
 * @version $Id MachineKey 2014-03-27 14:03:30z dungvnguyen $
 * @since 1.0
 */
final class MachineKey extends UniqueKey {
  //~ class properties ========================================================
  private final int time;
  private final int machine;
  private final int inc;

  //~ class members ===========================================================
  MachineKey(int time, int machine, int inc) {
    this.time = time;
    this.machine = machine;
    this.inc = inc;
  }


  @Override
  public byte[] bytes() {
    byte[] result = new byte[12];

    ByteBuffer buffer = ByteBuffer.wrap(result);
    buffer.order(ByteOrder.BIG_ENDIAN);
    buffer.putInt(time);
    buffer.putInt(machine);
    buffer.putInt(inc);

    return result;
  }

  @Override
  public int size() {
    return 12;
  }

  @Override
  public int compareTo(UniqueKey uniqueKey) {
    if (uniqueKey == null) return -1;

    int result = (uniqueKey.size() > 12 ? -1 : (uniqueKey.size() < 12 ? 1 : 0));
    if (result != 0) {
      return result;
    } else {
      byte[] obytes = uniqueKey.bytes();
      byte[] bytes = bytes();
      int first, second;
      for (int index = 0; index < 12; index++) {
        first = 0x0FF;
        first = first & bytes[index];
        second = 0x0FF;
        second = second & obytes[index];

        int diff = first - second;
        if (diff != 0) return diff;
      }

      // two key are the same.
      return 0;
    }
  }

  @Override
  public int hashCode() {
    return Arrays.hashCode(new int[]{time, machine, inc});
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    else if (o == null || !(o instanceof MachineKey)) return false;
    else {
      MachineKey that = (MachineKey) o;
      return (time == that.time && machine == that.machine && inc == that.inc);
    }
  }

  @Override
  public String toString() {
    return Encoders.bytesToHex(bytes());
  }

  static MachineKey from(byte[] source) throws UniqueKeyImportException {
    if (source == null || source.length != 12) {
      throw new UniqueKeyImportException("The key is invalid. It must not be null, and contains exactly 12 bytes.");
    }

    // create the buffer.
    ByteBuffer bb = ByteBuffer.wrap(source);
    bb.order(ByteOrder.BIG_ENDIAN);
    return new MachineKey(bb.getInt(), bb.getInt(), bb.getInt());
  }
}
