package org.qas.api.internal.util.google.hash;

import static org.qas.api.internal.util.google.base.Preconditions.checkNotNull;
import static org.qas.api.internal.util.google.base.Preconditions.checkPositionIndexes;
import org.qas.api.internal.util.google.primitives.Ints;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;

/**
 * AbstractByteHasher
 *
 * @author Dzung Nguyen
 * @version $Id AbstractByteHasher 2014-03-27 14:59:30z dungvnguyen $
 * @since 1.0
 */
abstract class AbstractByteHasher extends AbstractHasher {
  //~ class properties ========================================================
  private final ByteBuffer scratch = ByteBuffer.allocate(8).order(ByteOrder.LITTLE_ENDIAN);

  //~ class members ===========================================================
  /**
   * Updates this hasher with the given byte.
   */
  protected abstract void update(byte b);

  /**
   * Updates this hasher with the given bytes.
   */
  protected void update(byte[] b) {
    update(b, 0, b.length);
  }

  /**
   * Updates this hasher with {@code len} bytes starting at {@code off} in the given buffer.
   */
  protected void update(byte[] b, int off, int len) {
    for (int i = off; i < off + len; i++) {
      update(b[i]);
    }
  }

  @Override
  public Hasher putByte(byte b) {
    update(b);
    return this;
  }

  @Override
  public Hasher putBytes(byte[] bytes) {
    checkNotNull(bytes);
    update(bytes);
    return this;
  }

  @Override
  public Hasher putBytes(byte[] bytes, int off, int len) {
    checkPositionIndexes(off, off + len, bytes.length);
    update(bytes, off, len);
    return this;
  }

  /**
   * Updates the sink with the given number of bytes from the buffer.
   */
  private Hasher update(int bytes) {
    try {
      update(scratch.array(), 0, bytes);
    } finally {
      scratch.clear();
    }
    return this;
  }

  @Override
  public Hasher putShort(short s) {
    scratch.putShort(s);
    return update((Short.SIZE / Byte.SIZE));
  }

  @Override
  public Hasher putInt(int i) {
    scratch.putInt(i);
    return update(Ints.BYTES);
  }

  @Override
  public Hasher putLong(long l) {
    scratch.putLong(l);
    return update((Long.SIZE / Byte.SIZE));
  }

  @Override
  public Hasher putChar(char c) {
    scratch.putChar(c);
    return update((Character.SIZE / Byte.SIZE));
  }

  @Override
  public <T> Hasher putObject(T instance, Funnel<? super T> funnel) {
    funnel.funnel(instance, this);
    return this;
  }
}
