/*
 * Copyright (C) 2012 The Guava Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.qas.api.internal.util.google.collect;

/**
 * An {@link ImmutableAsList} implementation specialized for when the delegate collection is
 * already backed by an {@code ImmutableList} or array.
 *
 * @author Louis Wasserman
 * @author Dzung Nguyen
 * @version $Id RegularImmutableAsList 2014-03-26 22:42:30z dungvnguyen $
 * @since 1.0
 */
class RegularImmutableAsList<E> extends ImmutableAsList<E> {
  //~ class properties ========================================================
  private final ImmutableCollection<E> delegate;
  private final ImmutableList<? extends E> delegateList;

  //~ class members ===========================================================
  RegularImmutableAsList(ImmutableCollection<E> delegate, ImmutableList<? extends E> delegateList) {
    this.delegate = delegate;
    this.delegateList = delegateList;
  }

  RegularImmutableAsList(ImmutableCollection<E> delegate, Object[] array) {
    this(delegate, ImmutableList.asImmutableList(array));
  }

  @Override
  ImmutableCollection<E> delegateCollection() {
    return delegate;
  }

  ImmutableList<? extends E> delegateList() {
    return delegateList;
  }

  @SuppressWarnings("unchecked")  // safe covariant cast!
  @Override
  public UnmodifiableListIterator<E> listIterator(int index) {
    return (UnmodifiableListIterator<E>) delegateList.listIterator(index);
  }

  @Override
  int copyIntoArray(Object[] dst, int offset) {
    return delegateList.copyIntoArray(dst, offset);
  }

  @Override
  public E get(int index) {
    return delegateList.get(index);
  }
}
