/*
 * Copyright (C) 2011 The Guava Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.qas.api.internal.util.google.base;

import static org.qas.api.internal.util.google.base.Preconditions.checkNotNull;

import java.util.Collections;
import java.util.Set;

/**
 * Implementation of an {@link Optional} not containing a reference.
 *
 * @author Dzung Nguyen
 * @version $Id Absent 2014-03-26 18:55:30z dungvnguyen $
 * @since 1.0
 */
final class Absent<T> extends Optional<T> {
  //~ class properties ========================================================
  private static final long serialVersionUID = 0;
  static final Absent<Object> INSTANCE = new Absent<Object>();

  //~ class members ===========================================================
  private Absent() {}

  @SuppressWarnings("unchecked") // implementation is "fully variant"
  static <T> Optional<T> withType() {
    return (Optional<T>) INSTANCE;
  }

  @Override public boolean isPresent() {
    return false;
  }

  @Override public T get() {
    throw new IllegalStateException("Optional.get() cannot be called on an absent value");
  }

  @Override public T or(T defaultValue) {
    return checkNotNull(defaultValue, "use Optional.orNull() instead of Optional.or(null)");
  }

  @SuppressWarnings("unchecked") // safe covariant cast
  @Override public Optional<T> or(Optional<? extends T> secondChoice) {
    return (Optional<T>) checkNotNull(secondChoice);
  }

  @Override public T or(Supplier<? extends T> supplier) {
    return checkNotNull(supplier.get(),
        "use Optional.orNull() instead of a Supplier that returns null");
  }

  @Override public T orNull() {
    return null;
  }

  @Override
  public Set<T> asSet() {
    return Collections.emptySet();
  }

  @Override public <V> Optional<V> transform(Function<? super T, V> function) {
    checkNotNull(function);
    return Optional.absent();
  }

  @Override public boolean equals(Object object) {
    return object == this;
  }

  @Override public int hashCode() {
    return 0x598df91c;
  }

  @Override public String toString() {
    return "Optional.absent()";
  }

  private Object readResolve() {
    return INSTANCE;
  }
}
