package org.qas.api.internal.util;

import org.qas.api.internal.util.google.base.Strings;

import java.io.InputStream;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Versions
 *
 * @author: Dzung Nguyen
 * @version: $Id Versions 2014-03-26 11:02:30z dungvnguyen $
 * @since 1.0
 */
public final class Versions {
  //~ class properties ========================================================
  private static final Logger LOG = Logger.getLogger(Versions.class.getName());

  private static final String VERSION_FILE = "org/qas/api/version.properties";
  private static String version = null;
  private static String platform = null;
  private static String userAgent = null;

  //~ class members ===========================================================
  private Versions(){
    throw new AssertionError("Versions utilities class must not be instantiate!");
  }

  /**
   * @return the current platform of the http-auth-client.
   */
  public static String getPlatform() {
    if (Strings.isNullOrEmpty(platform)) {
      initializeVersion();
    }

    return platform;
  }

  /**
   * @return the current version of the http-auth-client.
   */
  public static String getVersion() {
    if (Strings.isNullOrEmpty(platform)) {
      initializeVersion();
    }

    return version;
  }

  /**
   * @return the current user-agent of the http-auth-client.
   */
  public static String getUserAgent() {
    if (Strings.isNullOrEmpty(userAgent)) {
      initializeUserAgent();
    }

    return userAgent;
  }

  /**
   * Load the version information from version.properties file.
   */
  private static void initializeVersion() {
    InputStream stream = Versions.class.getClassLoader().getResourceAsStream(VERSION_FILE);
    Properties versionProperties = new Properties();

    try {
      if (stream == null) {
       throw new Exception(VERSION_FILE + " not found on classpath");
        }
      versionProperties.load(stream);
      version = versionProperties.getProperty("version", "unknown");
      platform = versionProperties.getProperty("platform", "java");
    } catch (Exception ex) {
      if (LOG.isLoggable(Level.INFO))
        LOG.info("Unable to load version information with message: " + ex.getMessage());

      version = "unknown";
      platform = "java";
    }
  }

  /**
   * Load the version information and rebuild the user-agent information.
   */
  private static void initializeUserAgent() {
    StringBuffer buffer = new StringBuffer(1024);

    buffer.append("http-auth-client-").append(Versions.getPlatform().toLowerCase()).append("/")
        .append(Versions.getVersion()).append(" ")
        .append(System.getProperty("os.name").replace(' ', '_')).append("/")
        .append(System.getProperty("os.version").replace(' ', '_')).append(" ")
        .append(System.getProperty("java.vm.name").replace(' ', '_')).append("/")
        .append(System.getProperty("java.vm.version").replace(' ', '_'));

    // get the user language and region.
    String language = System.getProperty("user.language");
    String region = System.getProperty("user.region");

    if (!Strings.isNullOrEmpty(language) && !Strings.isNullOrEmpty(region)) {
      buffer.append(' ').append(language.replace(' ', '_')).append('_')
          .append(region.replace(' ', '_'));
    } else {
      //if (LOG.isLoggable(Level.INFO))
        //LOG.info("Unable to load user region information.");
    }

    userAgent = buffer.toString();
  }
}
