package org.qas.api;

import java.io.IOException;
import java.io.PrintStream;
import java.io.PrintWriter;

/**
 * AuthClientException
 *
 * @author: Dzung Nguyen
 * @version: $Id AuthClientException 2014-01-15 15:22:30z dungvnguyen $
 * @since 1.0
 */
public class AuthClientException extends RuntimeException {
  //~ class properties ========================================================
  private static final long serialVersionUID = 9035972689008422847L;

  // the new line.
  protected static final String NL = System.getProperty("line.separator");

  //~ class members ===========================================================

  /**
   * Creates {@link AuthClientException} object from the given exception
   * message.
   *
   * @param msg the given exception message.
   */
  public AuthClientException(String msg) {
    super(msg);
  }

  /**
   * Creates {@link AuthClientException} object from the given exception
   * message and the cause of exception.
   *
   * @param msg   the given exception message.
   * @param cause the given cause of exception.
   */
  public AuthClientException(String msg, Throwable cause) {
    super(msg, cause);
  }

  @Override
  public void printStackTrace() {
    System.err.println(this);
  }

  @Override
  public void printStackTrace(PrintStream s) {
    s.println(this);
  }

  @Override
  public void printStackTrace(PrintWriter s) {
    s.println(this);
  }

  @Override
  public String toString() {
    // create builder.
    StringBuilder builder = new StringBuilder();

    // build exception value.
    for (Throwable e = this; e != null; e = e.getCause()) {
      builder.append(NL);

      builder.append(e == this ? "Exception:" : "Caused By:").append(NL)
        .append("----------").append(NL);

      // check exception is error.
      if (e instanceof AssertionError
        || e instanceof IllegalArgumentException
        || e instanceof IllegalStateException
        || e instanceof NullPointerException) {
        builder.append("**************************************************************").append(NL)
          .append("**                                                          **").append(NL)
          .append("**   PLEASE SEND THIS STACK TRACE TO SUPPORT@QASYMPHONY.VN  **").append(NL)
          .append("**           HELP US FIX BUGS OR IMPROVE PRODUCT            **").append(NL)
          .append("**                                                          **").append(NL)
          .append("**************************************************************").append(NL);
      }

      // append exception information.
      try {
        appendExceptionTo(builder, e);
      } catch (IOException ioex) {
        throw new AssertionError(ioex);
      }

      // show stack trace information.
      StackTraceElement[] elements = e.getStackTrace();
      if (elements != null && elements.length > 0) {
        builder.append(">>> Stack trace: ").append(NL);

        for (StackTraceElement element : elements) {
          builder.append(">>>        at ")
            .append(element.getClassName())
            .append('.')
            .append(element.getMethodName())
            .append('(')
            .append(element.getFileName())
            .append(':')
            .append(Math.max(1, element.getLineNumber()))
            .append(')')
            .append(NL);
        }
      }
    }

    // return document.
    return builder.toString();
  }

  /**
   * Appends the exception information to appendable object.
   *
   * @param out the specified appenable to append exception information to.
   * @param e   Throwable
   * @throws IOException if
   */
  protected void appendExceptionTo(Appendable out, Throwable e) throws IOException {
    out.append(">>> Type: ").append(e.getClass().getName()).append(NL);
    out.append(">>> Message: ").append(e.getMessage()).append(NL);
  }
}
