package com.pushpole.sdk.util;

import android.annotation.TargetApi;
import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.os.Build;
import android.provider.Settings;
import android.support.annotation.RequiresApi;
import android.widget.RemoteViews;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.PlainConstants;
import com.pushpole.sdk.R;
import com.pushpole.sdk.action.Action;
import com.pushpole.sdk.controller.controllers.NotificationController;
import com.pushpole.sdk.internal.exceptions.NotificationBuildFailed;
import com.pushpole.sdk.message.downstream.NotificationDownstreamMessage;
import com.pushpole.sdk.network.ImageDownloader;
import com.pushpole.sdk.task.TaskManager;
import com.pushpole.sdk.task.tasks.NotificationHandleTask;

import static com.pushpole.sdk.PlainConstants.NOTIFICATION_CONTENT;
import static com.pushpole.sdk.PlainConstants.USER_BIG_CONTENT;
import static com.pushpole.sdk.PlainConstants.USER_BIG_TITLE;
import static com.pushpole.sdk.PlainConstants.USER_BUTTONS;
import static com.pushpole.sdk.PlainConstants.USER_CONTENT;
import static com.pushpole.sdk.PlainConstants.USER_ICON_URL;
import static com.pushpole.sdk.PlainConstants.USER_IMG_URL;
import static com.pushpole.sdk.PlainConstants.USER_JSON;
import static com.pushpole.sdk.PlainConstants.USER_SUMMARY;
import static com.pushpole.sdk.PlainConstants.USER_TITLE;
import static com.pushpole.sdk.util.Utility.isValidWebUrl;

/***
 * A class to build pushpole_custom_notification message for android device with api >= 16
 */
public class NotificationBuilderApi16AndAbove extends NotificationBuilder {
    private Context mContext;
    private NotificationDownstreamMessage mMessage;
    private int mNotificationId;

    /***
     * constructor
     *
     * @param context        app context
     * @param message        downstream message
     * @param notificationId unique ID
     */
    public NotificationBuilderApi16AndAbove(Context context, NotificationDownstreamMessage message, int notificationId) {
        mContext = context;
        mMessage = message;
        mNotificationId = notificationId;
    }

    /***
     * Build pushpole_custom_notification object
     */
    @TargetApi(21)
    public Notification build()
            throws NotificationBuildFailed {
        Notification.Builder builder = new Notification.Builder(mContext);
        setNotificationChannel(builder);
        setDefaultAttributes(builder);
        setNotificationContent(builder);
        setNotificationIcon(builder);
        setNotificationLed(builder);
        setNotificationButtons(builder);
        setNotificationSound(builder);
        builder.setOngoing(mMessage.isPermanentPush());
        //make push permanent if asked by user

        //builder.setPriority(getMsgPriority()); //shows error message

        switch (mMessage.getPriority()) {
            case Notification.PRIORITY_DEFAULT:
                builder.setPriority(Notification.PRIORITY_DEFAULT);
                break;

            case Notification.PRIORITY_HIGH:
                builder.setPriority(Notification.PRIORITY_HIGH);
                break;

            case Notification.PRIORITY_LOW:
                builder.setPriority(Notification.PRIORITY_LOW);
                break;

            case Notification.PRIORITY_MAX:
                builder.setPriority(Notification.PRIORITY_MAX);
                break;

            case Notification.PRIORITY_MIN:
                builder.setPriority(Notification.PRIORITY_MIN);
                break;

            default:
                builder.setPriority(Notification.PRIORITY_MAX);
                break;
        }

        return builder.build();
    }

    /***
     *         led on  led off
             * *******|*******|
     sound   *    A   |    B  |
     off     *        |       |
             * *******|*******|
      sound  *   C    |   D   |
      on     *        |       |
             * ***************
     * because led and sound setting of notification channels can't change after creating, we should create channels for different states.
     * when there is led color, we should create channel ids for each color code, so we've considered led color in notification ids
     * @param builder
     */

    @TargetApi(26)
    private void setNotificationChannel(Notification.Builder builder) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            if (mMessage.getNotifChannelID() != null && !mMessage.getNotifChannelID().isEmpty()) {
                builder.setChannelId(mMessage.getNotifChannelID());
            } else if (mMessage.hasValidSoundToPlay() && mMessage.getLedColor() != 0) { /** C state */
                builder.setChannelId(PlainConstants.DEFAULT_SILENT_CHANNEL_ID + mMessage.getLedColor());
                createNotificationChannel(PlainConstants.DEFAULT_SILENT_CHANNEL_ID + mMessage.getLedColor(), PlainConstants.DEFAULT_SILENT_CHANNEL_NAME, true, mMessage.getLedColor());

            } else if (mMessage.hasValidSoundToPlay() && mMessage.getLedColor() == 0) { /** D state*/
                builder.setChannelId(PlainConstants.DEFAULT_SILENT_CHANNEL_ID);
                createNotificationChannel(PlainConstants.DEFAULT_SILENT_CHANNEL_ID, PlainConstants.DEFAULT_SILENT_CHANNEL_NAME, true, 0);

            } else if (!mMessage.hasValidSoundToPlay() && mMessage.getLedColor() != 0) { /** A state*/
                builder.setChannelId(PlainConstants.NOTIF_DEFAULT_CHANNEL_ID + mMessage.getLedColor());
                createNotificationChannel(PlainConstants.NOTIF_DEFAULT_CHANNEL_ID + mMessage.getLedColor(), Constants.getVal(Constants.NOTIF_DEFAULT_CHANNEL_NAME), false, mMessage.getLedColor());

            } else if (!mMessage.hasValidSoundToPlay() && mMessage.getLedColor() == 0) {/** B state*/
                builder.setChannelId(PlainConstants.NOTIF_DEFAULT_CHANNEL_ID);
                createNotificationChannel(PlainConstants.NOTIF_DEFAULT_CHANNEL_ID, Constants.getVal(Constants.NOTIF_DEFAULT_CHANNEL_NAME), false, 0);

            }
        }
    }


    /***
     * set default attribute of pushpole_custom_notification
     */
    @TargetApi(21)
    private void setDefaultAttributes(Notification.Builder builder) {
        PendingIntent actionPendingIntent =
                PendingIntent.getService(mContext, IdGenerator.generateIntegerId(),
                        createActionIntent(mMessage.getAction(), null),
                        PendingIntent.FLAG_UPDATE_CURRENT);

        PendingIntent dismissPendingIntent =
                PendingIntent.getService(mContext, IdGenerator.generateIntegerId(),
                        createDismissIntent(), PendingIntent.FLAG_UPDATE_CURRENT);

        builder
                .setContentIntent(actionPendingIntent)
                .setDeleteIntent(dismissPendingIntent);
    }

    /***
     * set pushpole_custom_notification content
     */
    @TargetApi(21)
    private void setNotificationContent(Notification.Builder builder) throws NotificationBuildFailed {
        boolean isJustImgBkground = (mMessage.getJustImageUrl() != null && !mMessage.getJustImageUrl().isEmpty());
        if (isJustImgBkground) { //only show an image in background of notification
            setCustomLayout(builder);
        } else {
            builder.setContentTitle(mMessage.getTitle());
            builder.setContentText(mMessage.getContent());
        }

        if (mMessage.getTicker() != null && !mMessage.getTicker().isEmpty()) {
            builder.setTicker(mMessage.getTicker());
        }

        Notification.Style style = null;
        if (mMessage.getImageUrl() != null && !mMessage.getImageUrl().isEmpty()) {
            style = buildPictureImageStyle();
        } else if ((mMessage.getBigTitle() != null && !mMessage.getBigTitle().isEmpty()) ||
                (mMessage.getBigContent() != null && !mMessage.getBigContent().isEmpty())) {
            style = buildBigTextStyle();
        }

        if (style != null) {
            builder.setStyle(style);
        }
    }

    /***
     * build pushpole_custom_notification bigText, bigContentTitle and setSummaryText
     */
    @TargetApi(21)
    private Notification.Style buildBigTextStyle() {
        Notification.BigTextStyle style = new Notification.BigTextStyle();
        if (mMessage.getBigTitle() != null && !mMessage.getBigTitle().isEmpty()) {
            style.setBigContentTitle(mMessage.getBigTitle());
        } else {
            style.setBigContentTitle(mMessage.getTitle());
        }

        if (mMessage.getBigContent() != null && !mMessage.getBigContent().isEmpty()) {
            style.bigText(mMessage.getBigContent());
        } else {
            style.bigText(mMessage.getContent());
        }

        if (mMessage.getSummary() != null && !mMessage.getSummary().isEmpty()) {
            style.setSummaryText(mMessage.getSummary());
        }

        return style;
    }

    /***
     * build pushpole_custom_notification bigPicture and bigIcon and and bigContentTitle and setSummaryText
     */
    @TargetApi(21)
    private Notification.Style buildPictureImageStyle() throws NotificationBuildFailed {
        Notification.BigPictureStyle style = new Notification.BigPictureStyle();
        if (mMessage.getBigTitle() != null && !mMessage.getBigTitle().isEmpty()) {
            style.setBigContentTitle(mMessage.getBigTitle());
        }

        if (mMessage.getSummary() != null && !mMessage.getSummary().isEmpty()) {
            style.setSummaryText(mMessage.getSummary());
        }

        ImageDownloader downloader = new ImageDownloader(/*mContext*/);

        if (mMessage.getBigIconUrl() != null && !mMessage.getBigIconUrl().isEmpty()) {
            try {
                Bitmap bigIcon = downloader.downloadImage(mMessage.getBigIconUrl());
                if (bigIcon == null) {
                    throw new NotificationBuildFailed("Downloading pushpole_custom_notification big icon failed");
                }
                style.bigLargeIcon(bigIcon);
            } catch (SecurityException e) {
                android.util.Log.w("PushPole", "Could not download icon, probably due to missing INTERNET permission", e);
            }
        }

        if (mMessage.getImageUrl() != null && !mMessage.getImageUrl().isEmpty()) {
            try {
                Bitmap image = downloader.downloadImage(mMessage.getImageUrl());
                if (image == null) {
                    throw new NotificationBuildFailed("Downloading pushpole_custom_notification image failed");
                }
                style.bigPicture(image);
            } catch (SecurityException e) {
                android.util.Log.w("PushPole", "Could not download image, probably due to missing INTERNET permission", e);
            }
        }
        return style;
    }

    /***
     * set notification largeIcon
     */
    @TargetApi(21)
    private void setNotificationIcon(Notification.Builder builder) throws NotificationBuildFailed {
        boolean hasLargeIcon = (mMessage.getIconUrl() != null && !mMessage.getIconUrl().isEmpty());
        boolean usePushPoleIcon = mMessage.getUsePushPoleIcon();

        if (usePushPoleIcon) {
            builder.setSmallIcon(R.drawable.ic_pushpole);
        } else {
            if (mMessage.getNotifIcon() != null && !mMessage.getNotifIcon().isEmpty()) {//if user specified an icon from drawable as its notifIcon
                String mName = "ic_" + mMessage.getNotifIcon().replaceAll("[- ]", "_");
                int resId0 = mContext.getResources().getIdentifier(mName, "drawable", mContext.getPackageName());
                if (resId0 > 0) {
                    builder.setSmallIcon(resId0);
                }
            } else {
                boolean useWhiteIcon = (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP);
                int resId = mContext.getResources().getIdentifier("ic_silhouette", "drawable", mContext.getPackageName());
                if (useWhiteIcon && resId > 0) {
                    builder.setSmallIcon(resId);
                } else
                    builder.setSmallIcon(mContext.getApplicationInfo().icon);
            }
        }

        if (hasLargeIcon) {
            try {
                ImageDownloader downloader = new ImageDownloader(/*mContext*/);
                Bitmap image = downloader.downloadImage(mMessage.getIconUrl());

                if (image == null) {
                    throw new NotificationBuildFailed("Downloading pushpole_custom_notification large image icon failed");
                }
                builder.setLargeIcon(image);
            } catch (SecurityException e) {
                android.util.Log.w("PushPole", "Could not retrieve icon, probably due to missing INTERNET permission", e);
            }
        }
    }

    /**
     * set notification LED color
     *
     * @param builder notification builder
     */
    @TargetApi(21)
    private void setNotificationLed(Notification.Builder builder) {
        if (mMessage.getLedColor() != 0) {
            builder.setLights(mMessage.getLedColor(), mMessage.getLedOnTime(), mMessage.getLedOffTime());
        }
    }

    @RequiresApi(Build.VERSION_CODES.O)
    private void createNotificationChannel(String channelId, String channelName, boolean isSilent, int ledColor) {
        NotificationManager notificationManager = (NotificationManager) mContext.getSystemService(Context.NOTIFICATION_SERVICE);

        if (notificationManager.getNotificationChannel(channelId) == null) {
            NotificationChannel channel = new NotificationChannel(channelId,
                    channelName, NotificationManager.IMPORTANCE_HIGH);

            if (isSilent)
                channel.setSound(null, null);

            if (ledColor != 0) {
                channel.enableLights(true);
                channel.setLightColor(ledColor);
            }
            notificationManager.createNotificationChannel(channel);
        }
    }

    /**
     * set notification buttons
     *
     * @param builder notification builder
     */
    @TargetApi(21)
    private void setNotificationButtons(Notification.Builder builder) {
        if (mMessage.getButtons() == null) {
            return;
        }

        for (NotificationDownstreamMessage.Button button : mMessage.getButtons()) {
            int iconId = MaterialIconHelper.getIconResourceByMaterialName(mContext, button.getIcon());
            if (iconId == 0) {//TODO: PU-8
                //iconId = R.drawable.ic_empty; //This approach don't works when lib is pro-guarded
                iconId = mContext.getResources().getIdentifier("ic_empty", "drawable", mContext.getPackageName());
            }
            PendingIntent pendingIntent = PendingIntent.getService(mContext, IdGenerator.generateIntegerId(),
                    createActionIntent(button.getAction(), button.toPack()), 0);
            builder.addAction(iconId, button.getText(), pendingIntent);
        }
    }

    /***
     * set pushpole_custom_notification sound,
     * NO_SOUND
     * DEFAULT_SOUND
     * download sound from server when soundUrl available. In the case of failure default sound will be set
     *
     * @param builder pushpole_custom_notification builder
     */
    @TargetApi(21)
    private void setNotificationSound(Notification.Builder builder) {
        if (!isValidWebUrl(mMessage.getSoundUrl())) {
            builder.setSound(Settings.System.DEFAULT_NOTIFICATION_URI);
        }
    }

    @TargetApi(21)
    private void setCustomLayout(Notification.Builder builder) throws NotificationBuildFailed {
        RemoteViews contentView = new RemoteViews(mContext.getPackageName(), R.layout.pushpole_custom_notification);
        ImageDownloader downloader = new ImageDownloader(/*mContext*/);
        Bitmap bckgroundImg = downloader.downloadImage(mMessage.getJustImageUrl());
        if (bckgroundImg == null) {
            throw new NotificationBuildFailed("Downloading pushpole_custom_notification big icon failed");
        }
        contentView.setImageViewBitmap(R.id.pushpole_notif_bkgrnd_image, bckgroundImg);
        builder.setContent(contentView);
    }


    /***
     * create an intent for showing pushpole_custom_notification
     *
     * @param action intent
     * @param button  holds the clicked button
     * @return intent
     */
    private Intent createActionIntent(Action action, Pack button) {
        Pack data = new Pack();
        if (mMessage.getAction() != null) {
            data.putPack(Constants.getVal(Constants.F_ACTION), action.toPack());
        }
        data.putString(Constants.getVal(Constants.NOTIFICATION_ID), String.valueOf(mNotificationId));
        data.putString(Constants.getVal(Constants.F_ORIGINAL_MESSAGE_ID), mMessage.getMessageId());
        data.putString(Constants.getVal(Constants.F_RESPONSE_ACTION),
                Constants.getVal(Constants.RESPONSE_ACTION_CLICK));
        data.putPack(Constants.getVal(Constants.F_RESPONSE_BUTTON_ID), button);
        data.putPack(NOTIFICATION_CONTENT, notificationContent());
        return TaskManager.getInstance(mContext).createTaskIntent(
                NotificationHandleTask.class, data);
    }

    /***
     * Creat an intent to dismiss pushpole_custom_notification
     *
     * @return intent
     */
    private Intent createDismissIntent() {
        Pack data = new Pack();
        data.putString(Constants.getVal(Constants.NOTIFICATION_ID), String.valueOf(mNotificationId));
        data.putString(Constants.getVal(Constants.F_ORIGINAL_MESSAGE_ID), mMessage.getMessageId());
        data.putString(Constants.getVal(Constants.F_RESPONSE_ACTION),
                Constants.getVal(Constants.RESPONSE_ACTION_DISMISS));
        data.putPack(NOTIFICATION_CONTENT, notificationContent());
        return TaskManager.getInstance(mContext).createTaskIntent(
                NotificationHandleTask.class, data);
    }

    /**
     * Extract notification content that needs to be returned to the user.
     *
     * @return a pack of the notification data that holds {title, content, bigTitle, bigContent, summary, customContent and buttons}.
     */
    private Pack notificationContent() {
        Pack data = new Pack();
        // Include message content
        data.putString(USER_TITLE, mMessage.getTitle());
        data.putString(USER_CONTENT, mMessage.getContent());
        if (mMessage.getBigTitle() != null)
            data.putString(USER_BIG_TITLE, mMessage.getBigTitle());
        if (mMessage.getBigContent() != null)
            data.putString(USER_BIG_CONTENT, mMessage.getBigContent());
        if (mMessage.getSummary() != null)
            data.putString(USER_SUMMARY, mMessage.getSummary());
        if (mMessage.getCustomContent() != null)
            data.putString(USER_JSON, mMessage.getCustomContent().toJson());
        if (mMessage.getIconUrl() != null)
            data.putString(USER_ICON_URL, mMessage.getIconUrl());
        if (mMessage.getImageUrl() != null)
            data.putString(USER_IMG_URL, mMessage.getImageUrl());
        if (mMessage.getButtons() != null) {
            ListPack buttonData = new ListPack();
            for (NotificationDownstreamMessage.Button b : mMessage.getButtons()) {
                buttonData.add(b.toPack());
            }
            data.putListPack(USER_BUTTONS, buttonData);
        }
        return data;
    }

}
