package com.pushpole.sdk.task.tasks;

import android.content.Context;
import android.util.Log;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.SenderInfo;
import com.pushpole.sdk.controller.controllers.RegisterController;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.message.upstream.RegisterUpstreamMessage;
import com.pushpole.sdk.message.upstream.UpstreamMessageFactory;
import com.pushpole.sdk.network.UpstreamSender;
import com.pushpole.sdk.task.Result;
import com.pushpole.sdk.task.PushPoleTask;
import com.pushpole.sdk.task.options.SingletonTask;
import com.pushpole.sdk.util.Pack;

/***
 * A class that represent task for registering to pushpole-server
 */
@SingletonTask
public class ServerRegisterTask implements PushPoleTask {

    /***
     * The runTask contains operations for registering to pushpole-server
     * check {@code tokenState} and do appropriate operation
     *
     * @param context  the app context
     * @param taskPack the task pack
     * @return * @return {@code RESULT_SUCCESS} or {@code RESULT_RESCHEDULE} or {@code RESULT_FAILURE}
     */
    @Override
    public Result runTask(Context context, Pack taskPack) {
        int tokenState = SenderInfo.getInstance(context).getTokenState();
        RegisterController registerController = new RegisterController(context);
        switch (tokenState) {
            case SenderInfo.TOKEN_SYNCED:
                return Result.SUCCESS;

            case SenderInfo.NO_TOKEN:
                registerController.register();
                return Result.SUCCESS;

            case SenderInfo.TOKEN_RECEIVED:  //after successful gcm registration, token state updates to TOKEN_RECEIVED
                sendTokenSync(context);
                return Result.RESCHEDULE;

            default:
                Logger.error("Invalid value for Sender Info Token State: %d", tokenState);
                return Result.FAIL;
        }

    }

    /***
     * Send GCM token to pushpole-server, i.e. registering to PushPole server
     *
     * @param context
     */
    private void sendTokenSync(Context context) {
        String token = SenderInfo.getInstance(context).getToken();
        if(token == null){ //if somehow, token in KeyStore has become null, make token state 0 to get gcmToken again
            SenderInfo.getInstance(context).setTokenState(SenderInfo.NO_TOKEN);
            return;
        }

        RegisterUpstreamMessage.Factory messageFactory = new RegisterUpstreamMessage.Factory();
        RegisterUpstreamMessage message = messageFactory.buildUpstreamMessage(context);

        try {
            Log.i("PushPole", "Trying to register to PushPole");
            Pack regPack = new Pack();
            regPack.putPack(Constants.getVal(Constants.SERVER_REGISTER_T) ,message.toPack());
            regPack.putString(Constants.getVal(Constants.Info.F_GCM_TOKEN), token); //putting token in the root of registerMsg too
            regPack.putString(Constants.getVal(Constants.F_MESSAGE_ID), UpstreamMessageFactory.generateMsgId());

            new UpstreamSender(context).attemptSend(regPack);
        } catch (Exception e) {
            Logger.warning("Sending Upstream Message failed in ServerRegisterTask class - " + e.getLocalizedMessage());
            android.util.Log.e("PushPole", "Sending upstream message failed", e);
        }
    }
}
