package com.pushpole.sdk.network;


import android.content.Context;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.firebase.iid.FirebaseInstanceId;
import com.google.firebase.messaging.FirebaseMessaging;
import com.google.firebase.messaging.RemoteMessage;

import java.io.IOException;
import java.util.Map;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.FirebaseAppNotAvailableException;
import com.pushpole.sdk.PushPole;
import com.pushpole.sdk.PushPoleInfo;
import com.pushpole.sdk.SenderInfo;
import com.pushpole.sdk.controller.controllers.MobileCellInfoController;
import com.pushpole.sdk.controller.controllers.RegisterController;
import com.pushpole.sdk.device.DeviceIDHelper;
import com.pushpole.sdk.internal.db.NotifAndUpstreamMsgsDbOperation;
import com.pushpole.sdk.internal.log.LogData;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.internal.log.StatsCollector;
import com.pushpole.sdk.message.MessageStore;
import com.pushpole.sdk.message.upstream.UpstreamMessage;
import com.pushpole.sdk.task.TaskManager;
import com.pushpole.sdk.task.tasks.UpstreamSendTask;
import com.pushpole.sdk.util.Pack;
import com.pushpole.sdk.util.PackBundler;
import com.pushpole.sdk.util.PushPoleFailedException;

import static com.pushpole.sdk.Constants.F_ANDROID_ID;
import static com.pushpole.sdk.Constants.F_INSTANCE_ID;
import static com.pushpole.sdk.Constants.TIMESTAMP;

/***
 * A helper class for sending upstream message to pushpole-server
 */
public class UpstreamSender {
    private Context mContext;
    private MobileCellInfoController mobileCellInfoController;

    public UpstreamSender(Context context) {
        mContext = context;
    }

    /***
     * Send upstream message
     * in the case of failure, re-schedule sending upstream
     *
     * @param message the message
     */
    public void sendMessage(UpstreamMessage message) {
        try {
            attemptSend(message);
        } catch (IOException | PushPoleFailedException e) { //TODO: NoGoodLocationException: log level <= warning
            Logger.error("Sending Upstream Message failed in UpstreamSender.sendMessage() - " + e.getLocalizedMessage(), new LogData(
                    "Message ID", message.getMessageId(),
                    "Message", message.toPack().toJson())
            );
            rescheduleOnFail(message);
        }
    }

    /**
     * This method is used by {@link #attemptSend(Pack)} and is related to refactored upstream format
     * @param refactoredUpstreamMsg
     */
    public void sendMessage(Pack refactoredUpstreamMsg) {
        String msgId = refactoredUpstreamMsg.getString(Constants.getVal(Constants.F_MESSAGE_ID));
        try {
            attemptSend(refactoredUpstreamMsg);
        } catch (IOException | PushPoleFailedException | FirebaseAppNotAvailableException e) { //TODO: NoGoodLocationException: log level <= warning
            Logger.error("Sending Upstream Message failed in UpstreamSender.sendMessage() - " + e.getLocalizedMessage(), new LogData(
                    "Message ID", msgId,
                    "Message Type", "Refactored Upstream Message"));
            //sending upstream failed, reschedule its send
            MessageStore.getInstance().storeUpstreamMessage(mContext, refactoredUpstreamMsg, msgId);
            TaskManager.getInstance(mContext).scheduleTask(UpstreamSendTask.class, refactoredUpstreamMsg);
        }

    }

    private void rescheduleOnFail(UpstreamMessage msg) {
        MessageStore.getInstance().storeMessage(mContext, msg);
        Pack data = new Pack();
        data.putString(Constants.getVal(Constants.MESSAGE_ID), msg.getMessageId());
        TaskManager.getInstance(mContext).scheduleTask(UpstreamSendTask.class, data);
    }

    /***
     * Trying to get location if appropriate permission granted
     * send upstream to GCM
     *
     * @throws IOException
     */
    public void attemptSend(UpstreamMessage message) throws IOException/*gcm.send()*/,PushPoleFailedException {

        StatsCollector.increment(mContext, StatsCollector.STAT_SEND_ATTEMPTS);

        Pack dataPack = message.toPack();
        setUpstreamMessageDefaults(dataPack); //if location is null set default location values i.e. 0, 0
        Map<String, String> data = PackBundler.packToStringMap(dataPack);

        Logger.info("Sending Upstream Message", new LogData(
                "Message ID", message.getMessageId(),
                //"Message Type", message.getMessageType().toString(),
                "Data", data.toString(),
                "Size", String.valueOf(dataPack.toJson().length())
        ));


        FirebaseMessaging fm = FirebaseMessaging.getInstance();
        fm.send(new RemoteMessage.Builder(SenderInfo.getInstance(mContext).getSenderId() + "@gcm.googleapis.com")
                .setMessageId(message.getMessageId())
                .setData(data)
                .build());

        //saving this sent messages in DB
        NotifAndUpstreamMsgsDbOperation notifAndUpstreamMsgsDbOperation = NotifAndUpstreamMsgsDbOperation.getInstance(mContext);
        notifAndUpstreamMsgsDbOperation.open();
        if (!notifAndUpstreamMsgsDbOperation.isMsgExists(message.getMessageId())) {
            notifAndUpstreamMsgsDbOperation.insertMsg(message);
            Logger.debug("Sent Upstream Message stored in DB", new LogData(
                    "Message ID", message.getMessageId(),
                    //"Message Type", message.getMessageType().toString(),
                    "Size", String.valueOf(dataPack.toJson().length())));
        }
        StatsCollector.increment(mContext, StatsCollector.STAT_SENT_MESSAGES);
    }

    /**
     *  This method is used for sending upstream message with REFACTORED upstream format
     * @param dataPack
     * @throws PushPoleFailedException
     * @throws IOException
     */
    public void attemptSend(Pack dataPack) throws PushPoleFailedException, IOException, FirebaseAppNotAvailableException {

        StatsCollector.increment(mContext, StatsCollector.STAT_SEND_ATTEMPTS);
        setUpstreamMessageDefaults(dataPack);
        Map<String, String> data = PackBundler.packToStringMap(dataPack);
        String messageId = dataPack.getString(Constants.getVal(Constants.F_MESSAGE_ID));
                Logger.info("Sending Upstream Message", new LogData(
                        "Message ID", messageId,
                        "Message Type", UpstreamMessage.Type.REFACTORED_UPSTREAM.toString(),
                        "Data", data.toString(),
                        "Size", String.valueOf(dataPack.toJson().getBytes().length)
                ));

        SenderInfo senderInfo = SenderInfo.getInstance(mContext);

        FirebaseMessaging fm = PushPole.getFirebaseMessaging(mContext, senderInfo);

        fm.send(new RemoteMessage.Builder(senderInfo.getSenderId() + "@gcm.googleapis.com")
                .setMessageId(messageId)
                .setData(data)
                .build());

        //saving this sent messages in DB
        NotifAndUpstreamMsgsDbOperation notifAndUpstreamMsgsDbOperation = NotifAndUpstreamMsgsDbOperation.getInstance(mContext);
        notifAndUpstreamMsgsDbOperation.open();
        if (!notifAndUpstreamMsgsDbOperation.isMsgExists(messageId)) {
            notifAndUpstreamMsgsDbOperation.insertMsg(dataPack, messageId);
            Logger.debug("Sent Upstream Message stored in DB", new LogData(
                    "Message ID", messageId,
                    "Message Type ", "refactored upstream",
                    "Size", String.valueOf(dataPack.toJson().getBytes().length)));
        }
    }

    /***
     * set default values for upstream messages
     * set location latitude and longitude if available otherwise set them 0
     * set device ID, timestamp, instanceID, androidID
     *
     * @param pack     the data pack
     */
    private void setUpstreamMessageDefaults(Pack pack) {
        String instanceId = SenderInfo.getInstance(mContext).getInstanceId();
        if(instanceId == null){//reset token state to get token and instaceId agian.
            Logger.error("InstanceId is null, resetting token state to 0.");
            SenderInfo.getInstance(mContext).setTokenState(mContext, SenderInfo.NO_TOKEN);
            new RegisterController(mContext).register();
        }
        DeviceIDHelper deviceIDHelper = new DeviceIDHelper(mContext);

        pack.putString(Constants.getVal(F_INSTANCE_ID), SenderInfo.getInstance(mContext).getInstanceId());

        pack.putString(Constants.getVal(F_ANDROID_ID), deviceIDHelper.getAndroidId());
        pack.putString(Constants.getVal(TIMESTAMP), String.valueOf(System.currentTimeMillis())); // adds send-time to root of the upstream message
        if(mobileCellInfoController == null)
            mobileCellInfoController = new MobileCellInfoController(mContext);
        Pack connectedCell = mobileCellInfoController.getLocationOfConnectedCell();

        if (connectedCell != null && connectedCell.values().size() > 0) {
            pack.putPack(Constants.getVal(Constants.CELL_LOCATION_INFO), connectedCell);
        }

        try {
            String advertisingId = DeviceIDHelper.getAdvertisingId(mContext);
            pack.putString(Constants.getVal(Constants.GAID), advertisingId);
        } catch (Exception e) {
        }

        pack.putString(Constants.getVal(Constants.PUSHPOLE_VERSION_CODE), String.valueOf(PushPoleInfo.VERSION_CODE));
        pack.putString(Constants.getVal(Constants.APP_ID), SenderInfo.getInstance(mContext).getAppId());
    }
}
