package com.pushpole.sdk.message.downstream;

import java.util.ArrayList;
import java.util.List;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.topic.Topic;
import com.pushpole.sdk.util.ListPack;
import com.pushpole.sdk.util.Pack;

/***
 * A class that represent downstream message for update subscription
 */
public class UpdateSubscriptionsDownstreamMessage extends DownstreamMessage {
    private List<Topic> mSubscribeTo;
    private List<Topic> mUnsubscribeFrom;

    /**
     * return list of channels that must be subscribed
     *
     * @return {@link List} of topics
     */
    public List<Topic> getSubscribeTo() {
        return mSubscribeTo;
    }

    /***
     * set list of channels that must be subscribed
     *
     * @param subscribeTo {@link List} of topics
     */
    public void setSubscribeTo(List<Topic> subscribeTo) {
        mSubscribeTo = subscribeTo;
    }

    /***
     * return list of channels that must be unsubscribed
     *
     * @return {@link List} of topics
     */
    public List<Topic> getUnsubscribeFrom() {
        return mUnsubscribeFrom;
    }

    /***
     * set list of channels that must be unsubscribed
     *
     * @param unsubscribeFrom {@link List} of topics
     */
    public void setUnsubscribeFrom(List<Topic> unsubscribeFrom) {
        mUnsubscribeFrom = unsubscribeFrom;
    }

    /***
     * return message type
     *
     * @return {@code Type.UPDATE_SUBSCRIPTIONS}
     */
    @Override
    public DownstreamMessage.Type getMessageType() {
        return Type.UPDATE_SUBSCRIPTIONS;
    }

    /***
     * Factory class for generating {@link UpdateSubscriptionsDownstreamMessage} instance
     */
    public static class Factory extends DownstreamMessageFactory {

        /***
         * generate {@link UpdateSubscriptionsDownstreamMessage} instance
         *
         * @param data the data that contains list of channels for subscribing and unsubscribing
         * @return {@link UpdateSubscriptionsDownstreamMessage} instance
         */
        @Override
        public DownstreamMessage buildMessage(Pack data) {
            UpdateSubscriptionsDownstreamMessage message = new UpdateSubscriptionsDownstreamMessage();
            populateMessage(message, data);

            ListPack subListPack = data.getListPack(Constants.getVal(Constants.F_SUBSCRIBE_TO));
            ListPack unsubListPack = data.getListPack(Constants.getVal(Constants.F_UNSUBSCRIBE_FROM));

            List<Topic> subList = new ArrayList<>();
            List<Topic> unsubList = new ArrayList<>();

            if (subListPack != null) {
                for (int i = 0; i < subListPack.size(); i++) {
                    subList.add(new Topic(subListPack.getString(i)));
                }
            }

            if (unsubListPack != null) {
                for (int i = 0; i < unsubListPack.size(); i++) {
                    unsubList.add(new Topic(unsubListPack.getString(i)));
                }
            }

            message.setSubscribeTo(subList);
            message.setUnsubscribeFrom(unsubList);

            return message;
        }
    }
}
