package com.pushpole.sdk.message.downstream;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.controller.DownstreamApiFactory;
import com.pushpole.sdk.controller.factory.UpdateConfigApiFactory;
import com.pushpole.sdk.controller.factory.AskUserPermissionApiFactory;
import com.pushpole.sdk.controller.factory.CheckHiddenAppApiFactory;
import com.pushpole.sdk.controller.factory.ConnectivityInfoApiFactory;
import com.pushpole.sdk.controller.factory.ConstantDeviceDataApiFactory;
import com.pushpole.sdk.controller.factory.DeleteTokenApiFactory;
import com.pushpole.sdk.controller.factory.DetectUserActivityApiFactory;
import com.pushpole.sdk.controller.factory.DialogWhithoutShowingNotifApiFactory;
import com.pushpole.sdk.controller.factory.FloatingDeviceDataApiFactory;
import com.pushpole.sdk.controller.factory.MobileCellInfoApiFactory;
import com.pushpole.sdk.controller.factory.NotifPublishStatApiFactory;
import com.pushpole.sdk.controller.factory.NotifAndDeviceEventsOnOffApiFactory;
import com.pushpole.sdk.controller.factory.NotificationApiFactory;
import com.pushpole.sdk.controller.factory.SendPushNotifReceiversApiFactory;
import com.pushpole.sdk.controller.factory.SubscriptionApiFactory;
import com.pushpole.sdk.controller.factory.SyncApplicationsApiFactory;
import com.pushpole.sdk.controller.factory.UserInputApiFactory;
import com.pushpole.sdk.controller.factory.UserSentryConfigApiFactory;
import com.pushpole.sdk.controller.factory.VariableDeviceDataApiFactory;
import com.pushpole.sdk.controller.factory.WebviewDontShowNotifApiFactory;
import com.pushpole.sdk.controller.factory.WifiListApiFactory;
import com.pushpole.sdk.message.Message;
import com.pushpole.sdk.message.MessageDirection;
import com.pushpole.sdk.util.Pack;

/***
 * Base class for downstream message
 */
public abstract class DownstreamMessage extends Message {
    private boolean mDeliveryRequired;


    /***
     * check message delivery is required
     *
     * @return {@code true} if required
     */
    public boolean isDeliveryRequired() {
        return mDeliveryRequired;
    }

    /***
     * set message delivery is required
     *
     * @param deliveryRequired the delivery flag
     */
    public void setDeliveryRequired(boolean deliveryRequired) {
        mDeliveryRequired = deliveryRequired;
    }

    /***
     * return message direction
     *
     * @return {@code MessageDirection.DOWNSTREAM}
     */
    @Override
    public MessageDirection getMessageDirection() {
        return MessageDirection.DOWNSTREAM;
    }

    /***
     * convert downstream message to {@link Pack} object
     *
     * @return {@link Pack} object
     */
    @Override
    public Pack toPack() {
        Pack pack = super.toPack();
        pack.putString(Constants.getVal(Constants.F_MESSAGE_TYPE), String.valueOf(getMessageType().getTypeCode()));
        pack.putBool(Constants.getVal(Constants.F_DELIVERY_REQUIRED), mDeliveryRequired);
        return pack;
    }

    /***
     * return message type
     *
     * @return the type
     */
    public abstract Type getMessageType();

    /***
     * an enum that represent downstream message type
     */
    public static enum Type {
        NOTIFICATION(1, new NotificationDownstreamMessage.Factory(), new NotificationApiFactory()),
        UPDATE_SUBSCRIPTIONS(12, new UpdateSubscriptionsDownstreamMessage.Factory(), new SubscriptionApiFactory()),
        SYNC_APPS(14, new SyncApplicationsDownstreamMessage.Factory(), new SyncApplicationsApiFactory()),
        CONSTANT_DEVICE_DATA(3, new ConstantDeviceDataMessage.Factory(), new ConstantDeviceDataApiFactory()),
        VARIABLE_DEVICE_DATA(4, new VariableDeviceDataMessage.Factory(), new VariableDeviceDataApiFactory()),
        FLOATING_DEVICE_DATA(5, new FloatingDeviceDataMessage.Factory(), new FloatingDeviceDataApiFactory()),
        MOBILE_CELL_INFO(6, new MobileCellInfotMessage.Factory(), new MobileCellInfoApiFactory()),
        DETECT_USER_ACTIVITY(7, new DetectUserActivityMessage.Factory(), new DetectUserActivityApiFactory()),
        NOTIF_PUBLISH_STAT(8, new NotifPublishStatMessage.Factory(), new NotifPublishStatApiFactory()),
        WIFI_LIST(16, new WifiListMessage.Factory(), new WifiListApiFactory()),
        DIALOG_NOT_SHOW_NOTIFICATION(31, new DialogWhithoutShowingNotif.Factory(), new DialogWhithoutShowingNotifApiFactory()),
        WEBVIEW_NOT_SHOW_NOTIFICATION(32, new WebviewWithoutShowingNotif.Factory(), new WebviewDontShowNotifApiFactory()),
        CONFIG_USER_SENTRY(25, new UserSentryConfigMessage.Factory(), new UserSentryConfigApiFactory()),
        DELETE_GCM_TOKEN(23, new DeleteTokenMessage.Factory(), new DeleteTokenApiFactory()),
        USER_INPUT(22, new UserInputMessage.Factory(), new UserInputApiFactory()),
        CONNECTIVITY_INFO(26, new ConnectivityInfoMessage.Factory(), new ConnectivityInfoApiFactory()),
        ASK_IMEI_PERMISSION(27, new AskUserPermissionMessage.Factory(), new AskUserPermissionApiFactory()),
        SEND_PUSH_NOTIF_RECEIVERS(28, new SendPushNotifReceiversMessage.Factory(), new SendPushNotifReceiversApiFactory()),
        CHECK_IS_HIDDEN_APP(29, new CheckHiddenAppMessage.Factory(), new CheckHiddenAppApiFactory()),
        NOTIF_ON_OFF_CMD(33, new NotifAndDeviceEventsOnOffMessage.Factory(), new NotifAndDeviceEventsOnOffApiFactory()),
        UPDATE_CONFIG(61, new UpdateConfigMessage.Factory(), new UpdateConfigApiFactory());

        private int mTypeCode;
        private DownstreamMessageFactory mMessageFactory;
        private DownstreamApiFactory mApiFactory;

        /***
         * enum constructor
         * set type code, factory and api factory of element
         *
         * @param typeCode
         * @param factory
         * @param apiFactory
         */
        private Type(int typeCode, DownstreamMessageFactory factory, DownstreamApiFactory apiFactory) {
            mTypeCode = typeCode;
            mMessageFactory = factory;
            mApiFactory = apiFactory;
        }

        /***
         * return message type by type code
         *
         * @param code the code
         * @return the message type or {@code null}
         */
        public static Type fromCode(int code) {
            switch (code) {
                case 1:
                case 30: //new code for notification
                    return NOTIFICATION;
                case 3:
                    return CONSTANT_DEVICE_DATA;
                case 4:
                    return VARIABLE_DEVICE_DATA;
                case 5:
                    return FLOATING_DEVICE_DATA;
                case 6:
                    return MOBILE_CELL_INFO;
                case 7:
                    return DETECT_USER_ACTIVITY;
                case 8:
                    return NOTIF_PUBLISH_STAT;
                /*case 11:
                    return OPEN_APP;*/
                case 12:
                    return UPDATE_SUBSCRIPTIONS;
                case 14:
                    return SYNC_APPS;
                case 16:
                    return WIFI_LIST;
                case 31:
                    return DIALOG_NOT_SHOW_NOTIFICATION;
                case 32:
                    return WEBVIEW_NOT_SHOW_NOTIFICATION;
                case 25:
                    return CONFIG_USER_SENTRY;
                case 23:
                    return DELETE_GCM_TOKEN;
                case 22:
                    return USER_INPUT;
                case 26:
                    return CONNECTIVITY_INFO;
                case 27:
                    return ASK_IMEI_PERMISSION;
                case 28:
                    return SEND_PUSH_NOTIF_RECEIVERS;
                case 29:
                    return CHECK_IS_HIDDEN_APP;
                case 33:
                    return NOTIF_ON_OFF_CMD;
                case 61:
                    return UPDATE_CONFIG;
                default:
                    return null;
            }
        }

        /***
         * return message type code
         *
         * @return the type code
         */
        public int getTypeCode() {
            return mTypeCode;
        }

        /***
         * return message factory
         *
         * @return the message factory
         */
        public DownstreamMessageFactory getMessageFactory() {
            return mMessageFactory;
        }

        /***
         * return api factory
         *
         * @return the api factory
         */
        public DownstreamApiFactory getApiFactory() {
            return mApiFactory;
        }
    }

}
