package com.pushpole.sdk.device;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.util.Pack;

/**
 * Details and Information available for an application installed on the client device.
 *
 * @author Hadi Zolfaghari
 */
public class ApplicationDetails implements Comparable<ApplicationDetails>{
    private String mPackageName;
    private String mVersion;
    private long mFirstInstallTime;
    private long mLastUpdateTime;
    private String mInstallerPackageName;
    private String mAppName;
    private long mTimestamp;
    private boolean mIsAppHidden;

    /**
     * Creates an instance of {@code ApplicationDetails} from data represented as a {@link Pack}
     *
     * @param pack the serialized {@code ApplicationDetails} as a {@code Pack}
     * @return the {@code ApplicationDetails} created from the given data
     */
    public static ApplicationDetails fromPack(Pack pack) {
        ApplicationDetails app = new ApplicationDetails();
        app.setPackageName(pack.getString(Constants.getVal(Constants.F_APP_PACKAGENAME), null));
        app.setVersion(pack.getString(Constants.getVal(Constants.F_APP_VERSION), null));
        app.setInstallerPackageName(pack.getString(Constants.getVal(Constants.F_APP_INSTALLER), null));
        app.setAppName(pack.getString(Constants.getVal(Constants.F_APP_NAME), null));
        app.setIsAppHidden(pack.getBool(Constants.getVal(Constants.F_APP_IS_HIDDEN), false));
        try {
            app.setFirstInstallTime(Long.parseLong(pack.getString(Constants.getVal(Constants.F_APP_FIRST_INSTALL_TIME), "0")));
        } catch (NumberFormatException e) {
            app.setFirstInstallTime(0);
        }
        try {
            app.setLastUpdateTime(Long.parseLong(pack.getString(Constants.getVal(Constants.F_APP_LAST_UPDATE_TIME), "0")));
        } catch (NumberFormatException e) {
            app.setLastUpdateTime(0);
        }
        try{
            String timeStr = pack.getString(Constants.getVal(Constants.TIMESTAMP), "0");
            app.setTimestamp(Long.parseLong(timeStr));
        }catch (Exception e){
            app.setTimestamp(System.currentTimeMillis());
        }
        return app;
    }

    /**
     * @return the application package name
     */
    public String getPackageName() {
        return mPackageName;
    }

    /**
     * @param packageName the application's package name to set
     */
    public void setPackageName(String packageName) {
        mPackageName = packageName;
    }

    /**
     * @return the application version
     */
    public String getVersion() {
        return mVersion;
    }

    /**
     * @param version the application version
     */
    public void setVersion(String version) {
        mVersion = version;
    }

    /**
     * @return the time in milliseconds since epoch when the application was first installed
     */
    public long getFirstInstallTime() {
        return mFirstInstallTime;
    }

    /**
     * @param time the time in milliseconds since epoch when the application was first installed
     */
    public void setFirstInstallTime(long time) {
        mFirstInstallTime = time;
    }

    /**
     * @return the time in milliseconds since epoch when the application was last updated
     */
    public long getLastUpdateTime() {
        return mLastUpdateTime;
    }

    /**
     * @param time the time in milliseconds since epoch when the application was last updated
     */
    public void setLastUpdateTime(long time) {
        mLastUpdateTime = time;
    }

    /**
     * @return the package name of the market application through which this application was
     * installed from
     */
    public String getInstallerPackageName() {
        return mInstallerPackageName;
    }

    /**
     * @param installerPackageName the package name of the market application through which this
     *                             application was installed from
     */
    public void setInstallerPackageName(String installerPackageName) {
        mInstallerPackageName = installerPackageName;
    }

    public String getAppName() {
        return mAppName;
    }

    public void setAppName(String appName) {
        this.mAppName = appName;
    }

    public long getTimestamp() {
        return mTimestamp;
    }

    public void setTimestamp(long timestamp) {
        this.mTimestamp = timestamp;
    }

    public boolean isAppHidden() {
        return this.mIsAppHidden;
    }

    public void setIsAppHidden(boolean isAppHidden) {
        this.mIsAppHidden = isAppHidden;
    }

    /**
     * Serializes this {@code ApplicationDetails} instance into a {@link Pack}
     *
     * @return the {@code Pack} serialization of this {@code ApplicationDetails} instance
     */
    public Pack toPack() {
        Pack pack = new Pack();
        pack.putString(Constants.getVal(Constants.F_APP_PACKAGENAME), mPackageName);
        pack.putString(Constants.getVal(Constants.F_APP_VERSION), mVersion);
        if (mInstallerPackageName != null) {
            pack.putString(Constants.getVal(Constants.F_APP_INSTALLER), mInstallerPackageName);
        }
        if (mFirstInstallTime != 0) {
            pack.putString(Constants.getVal(Constants.F_APP_FIRST_INSTALL_TIME), String.valueOf(mFirstInstallTime));
        }
        if (mLastUpdateTime != 0) {
            pack.putString(Constants.getVal(Constants.F_APP_LAST_UPDATE_TIME), String.valueOf(mLastUpdateTime));
        }
        if(mAppName != null && !mAppName.equalsIgnoreCase("null"))
            pack.putString(Constants.getVal(Constants.F_APP_NAME), mAppName);
        pack.putString(Constants.getVal(Constants.TIMESTAMP), String.valueOf(System.currentTimeMillis()));
        pack.putString(Constants.getVal(Constants.APP_IS_HIDDEN), String.valueOf(mIsAppHidden));
        return pack;
    }

    @Override
    public int compareTo(ApplicationDetails another) {
        if(this.getFirstInstallTime() > another.getFirstInstallTime())
            return 1;
        if(this.getFirstInstallTime() < another.getFirstInstallTime())
            return -1;
        else
            return 0;
    }
}
