/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.util;

import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.Predicate;
import java.util.function.ToIntFunction;
import java.util.stream.Stream;
import org.eclipse.jetty.util.AtomicBiInteger;
import org.eclipse.jetty.util.Pool;
import org.eclipse.jetty.util.annotation.ManagedAttribute;
import org.eclipse.jetty.util.annotation.ManagedObject;
import org.eclipse.jetty.util.component.Dumpable;
import org.eclipse.jetty.util.component.DumpableCollection;
import org.eclipse.jetty.util.thread.AutoLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ManagedObject
public class ConcurrentPool<P>
implements Pool<P>,
Dumpable {
    public static final int OPTIMAL_MAX_SIZE = 256;
    private static final Logger LOG = LoggerFactory.getLogger(ConcurrentPool.class);
    private final List<Holder<P>> entries = new CopyOnWriteArrayList<Holder<P>>();
    private final int maxSize;
    private final StrategyType strategyType;
    private final AutoLock lock = new AutoLock();
    private final AtomicInteger nextIndex;
    private final ToIntFunction<P> maxMultiplex;
    private final LongAdder leaked = new LongAdder();
    private volatile boolean terminated;

    public ConcurrentPool(StrategyType strategyType, int maxSize) {
        this(strategyType, maxSize, pooled -> 1);
    }

    @Deprecated
    public ConcurrentPool(StrategyType strategyType, int maxSize, boolean cache) {
        this(strategyType, maxSize, pooled -> 1);
    }

    @Deprecated
    public ConcurrentPool(StrategyType strategyType, int maxSize, boolean cache, ToIntFunction<P> maxMultiplex) {
        this(strategyType, maxSize, maxMultiplex);
    }

    public ConcurrentPool(StrategyType strategyType, int maxSize, ToIntFunction<P> maxMultiplex) {
        if (maxSize > 256 && LOG.isDebugEnabled()) {
            LOG.debug("{} configured with max size {} which is above the recommended value {}", this.getClass().getSimpleName(), maxSize, 256);
        }
        this.maxSize = maxSize;
        this.strategyType = Objects.requireNonNull(strategyType);
        this.nextIndex = strategyType == StrategyType.ROUND_ROBIN ? new AtomicInteger() : null;
        this.maxMultiplex = Objects.requireNonNull(maxMultiplex);
    }

    @ManagedAttribute(value="number of entries leaked (not released nor referenced)")
    public long getLeaked() {
        return this.leaked.longValue();
    }

    private int getMaxMultiplex(P pooled) {
        return this.maxMultiplex.applyAsInt(pooled);
    }

    private void leaked(Holder<P> holder) {
        this.leaked.increment();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Leaked " + String.valueOf(holder));
        }
        this.leaked();
    }

    protected void leaked() {
    }

    @Override
    public Pool.Entry<P> reserve() {
        try (AutoLock ignored = this.lock.lock();){
            if (this.terminated) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("terminated, cannot reserve entry for {}", (Object)this);
                }
                Pool.Entry<P> entry = null;
                return entry;
            }
            int entriesSize = this.entries.size();
            if (this.maxSize > 0 && entriesSize >= this.maxSize) {
                this.sweep();
                entriesSize = this.entries.size();
                if (entriesSize >= this.maxSize) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("no space: {} >= {}, cannot reserve entry for {}", entriesSize, this.maxSize, this);
                    }
                    Pool.Entry<P> entry = null;
                    return entry;
                }
            }
            ConcurrentEntry entry = new ConcurrentEntry(this);
            this.entries.add(entry.getHolder());
            if (LOG.isDebugEnabled()) {
                LOG.debug("returning reserved entry {} for {}", (Object)entry, (Object)this);
            }
            ConcurrentEntry concurrentEntry = entry;
            return concurrentEntry;
        }
    }

    void sweep() {
        for (int i2 = 0; i2 < this.entries.size(); ++i2) {
            Holder<P> holder = this.entries.get(i2);
            if (holder.getEntry() != null) continue;
            this.entries.remove(i2--);
            this.leaked(holder);
        }
    }

    @Override
    public Pool.Entry<P> acquire() {
        if (this.terminated) {
            return null;
        }
        int size = this.entries.size();
        if (size == 0) {
            return null;
        }
        int index = this.startIndex(size);
        int tries = size;
        while (tries-- > 0) {
            try {
                Holder<P> holder = this.entries.get(index);
                if (holder != null) {
                    ConcurrentEntry entry = (ConcurrentEntry)holder.getEntry();
                    if (entry == null) {
                        this.entries.remove(index);
                        this.leaked(holder);
                        continue;
                    }
                    if (entry.tryAcquire()) {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("returning entry {} for {}", (Object)entry, (Object)this);
                        }
                        return entry;
                    }
                }
            }
            catch (IndexOutOfBoundsException e) {
                LOG.trace("IGNORED", e);
                size = this.entries.size();
                if (size == 0) break;
            }
            if (++index < size) continue;
            index = 0;
        }
        return null;
    }

    private int startIndex(int size) {
        return switch (this.strategyType.ordinal()) {
            default -> throw new IncompatibleClassChangeError();
            case 0 -> 0;
            case 1 -> ThreadLocalRandom.current().nextInt(size);
            case 3 -> this.nextIndex.getAndUpdate(c -> Math.max(0, c + 1)) % size;
            case 2 -> (int)(Thread.currentThread().getId() % (long)size);
        };
    }

    private boolean release(Pool.Entry<P> entry) {
        boolean released = ((ConcurrentEntry)entry).tryRelease();
        if (LOG.isDebugEnabled()) {
            LOG.debug("released {} {} for {}", released, entry, this);
        }
        return released;
    }

    private boolean remove(Pool.Entry<P> entry) {
        boolean removed = ((ConcurrentEntry)entry).tryRemove();
        if (LOG.isDebugEnabled()) {
            LOG.debug("removed {} {} for {}", removed, entry, this);
        }
        if (!removed) {
            return false;
        }
        Holder holder = ((ConcurrentEntry)entry).getHolder();
        boolean evicted = this.entries.remove(holder);
        if (LOG.isDebugEnabled()) {
            LOG.debug("evicted {} {} for {}", evicted, entry, this);
        }
        return true;
    }

    @Override
    public boolean isTerminated() {
        return this.terminated;
    }

    @Override
    public Collection<Pool.Entry<P>> terminate() {
        List<Pool.Entry<P>> copy;
        if (LOG.isDebugEnabled()) {
            LOG.debug("terminating {}", (Object)this);
        }
        try (AutoLock ignored = this.lock.lock();){
            this.terminated = true;
            copy = this.entries.stream().map(Holder::getEntry).filter(Objects::nonNull).toList();
            this.entries.clear();
        }
        copy.forEach(entry -> ((ConcurrentEntry)entry).terminate());
        return copy;
    }

    private boolean terminate(Pool.Entry<P> entry) {
        boolean terminated = ((ConcurrentEntry)entry).tryTerminate();
        if (!terminated && LOG.isDebugEnabled()) {
            LOG.debug("entry still in use or already terminated {} for {}", (Object)entry, (Object)this);
        }
        return terminated;
    }

    @Override
    public int size() {
        return this.entries.size();
    }

    @Override
    public int getMaxSize() {
        return this.maxSize;
    }

    @Override
    public Stream<Pool.Entry<P>> stream() {
        return this.entries.stream().map(Holder::getEntry).filter(Objects::nonNull);
    }

    @Override
    public int getReservedCount() {
        return this.getCount(Pool.Entry::isReserved);
    }

    @Override
    public int getIdleCount() {
        return this.getCount(Pool.Entry::isIdle);
    }

    @Override
    public int getInUseCount() {
        return this.getCount(Pool.Entry::isInUse);
    }

    @Override
    public int getTerminatedCount() {
        return this.getCount(Pool.Entry::isTerminated);
    }

    private int getCount(Predicate<Pool.Entry<P>> predicate) {
        int count = 0;
        for (Holder<P> holder : this.entries) {
            Pool.Entry<P> entry = holder.getEntry();
            if (entry == null || !predicate.test(entry)) continue;
            ++count;
        }
        return count;
    }

    @Override
    public void dump(Appendable out, String indent) throws IOException {
        Dumpable.dumpObjects(out, indent, this, new DumpableCollection("entries", this.entries));
    }

    public String toString() {
        return String.format("%s@%x[strategy=%s,inUse=%d,size=%d,max=%d,leaked=%d,terminated=%b]", new Object[]{this.getClass().getSimpleName(), this.hashCode(), this.strategyType, this.getInUseCount(), this.size(), this.getMaxSize(), this.getLeaked(), this.isTerminated()});
    }

    public static enum StrategyType {
        FIRST,
        RANDOM,
        THREAD_ID,
        ROUND_ROBIN;

    }

    public static class ConcurrentEntry<E>
    implements Pool.Entry<E> {
        private final AtomicBiInteger state = new AtomicBiInteger(0, -1);
        private final ConcurrentPool<E> pool;
        private final Holder<E> holder;
        private E pooled;

        public ConcurrentEntry(ConcurrentPool<E> pool) {
            this.pool = pool;
            this.holder = new Holder(this);
        }

        private Holder<E> getHolder() {
            return this.holder;
        }

        @Override
        public boolean enable(E pooled, boolean acquire) {
            Objects.requireNonNull(pooled);
            if (!this.isReserved()) {
                if (this.isTerminated()) {
                    return false;
                }
                throw new IllegalStateException("Entry already enabled " + String.valueOf(this) + " for " + String.valueOf(this.pool));
            }
            this.pooled = pooled;
            if (this.tryEnable(acquire)) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("enabled {} for {}", (Object)this, (Object)this.pool);
                }
                return true;
            }
            this.pooled = null;
            if (this.isTerminated()) {
                return false;
            }
            throw new IllegalStateException("Entry already enabled " + String.valueOf(this) + " for " + String.valueOf(this.pool));
        }

        @Override
        public E getPooled() {
            return this.pooled;
        }

        @Override
        public boolean release() {
            return this.pool.release(this);
        }

        @Override
        public boolean remove() {
            return this.pool.remove(this);
        }

        private boolean terminate() {
            return this.pool.terminate(this);
        }

        private boolean tryEnable(boolean acquire) {
            boolean enabled = this.state.compareAndSet(0, 0, -1, acquire ? 1 : 0);
            if (enabled && !acquire) {
                this.getHolder().released();
            }
            return enabled;
        }

        private boolean tryAcquire() {
            int newMultiplexCount;
            long encoded;
            do {
                if (AtomicBiInteger.getHi(encoded = this.state.get()) < 0) {
                    return false;
                }
                int multiplexCount = AtomicBiInteger.getLo(encoded);
                if (multiplexCount < 0) {
                    return false;
                }
                int maxMultiplexed = this.pool.getMaxMultiplex(this.pooled);
                if (maxMultiplexed > 0 && multiplexCount >= maxMultiplexed) {
                    return false;
                }
                newMultiplexCount = multiplexCount + 1;
                if (newMultiplexCount >= 0) continue;
                return false;
            } while (!this.state.compareAndSet(encoded, 0, newMultiplexCount));
            if (newMultiplexCount == 1) {
                this.getHolder().acquired();
            }
            return true;
        }

        private boolean tryRelease() {
            int multiplexCount;
            int newMultiplexCount;
            long encoded;
            do {
                if (AtomicBiInteger.getHi(encoded = this.state.get()) < 0) {
                    return false;
                }
                multiplexCount = AtomicBiInteger.getLo(encoded);
                if (multiplexCount > 0) continue;
                return false;
            } while (!this.state.compareAndSet(encoded, 0, newMultiplexCount = multiplexCount - 1));
            if (newMultiplexCount == 0) {
                this.getHolder().released();
            }
            return true;
        }

        private boolean tryRemove() {
            int multiplexCount;
            int newMultiplexCount;
            boolean result;
            int removed;
            long encoded;
            do {
                encoded = this.state.get();
                removed = AtomicBiInteger.getHi(encoded);
                multiplexCount = AtomicBiInteger.getLo(encoded);
                if (removed != -2) continue;
                return false;
            } while (!this.state.compareAndSet(encoded, removed = (result = (newMultiplexCount = multiplexCount <= 0 ? multiplexCount : multiplexCount - 1) <= 0) ? -2 : -1, newMultiplexCount));
            return result;
        }

        private boolean tryTerminate() {
            int multiplexCount;
            long encoded;
            do {
                if (AtomicBiInteger.getHi(encoded = this.state.get()) >= 0) continue;
                return false;
            } while (!this.state.compareAndSet(encoded, -1, multiplexCount = AtomicBiInteger.getLo(encoded)));
            return multiplexCount <= 0;
        }

        @Override
        public boolean isTerminated() {
            return this.state.getHi() < 0;
        }

        @Override
        public boolean isReserved() {
            return this.state.getLo() < 0;
        }

        @Override
        public boolean isIdle() {
            return this.state.getLo() == 0;
        }

        @Override
        public boolean isInUse() {
            return this.state.getLo() > 0;
        }

        public String toString() {
            long encoded = this.state.get();
            return String.format("%s@%x{terminated=%b,multiplex=%d,pooled=%s}", this.getClass().getSimpleName(), this.hashCode(), AtomicBiInteger.getHi(encoded) < 0, AtomicBiInteger.getLo(encoded), this.getPooled());
        }
    }

    private static class Holder<P> {
        private final WeakReference<ConcurrentEntry<P>> _weak;
        private volatile ConcurrentEntry<P> _strong;

        protected Holder(ConcurrentEntry<P> entry) {
            this._weak = new WeakReference<ConcurrentEntry<P>>(entry);
        }

        public Pool.Entry<P> getEntry() {
            return (Pool.Entry)this._weak.get();
        }

        public void released() {
            this._strong = (ConcurrentEntry)this._weak.get();
        }

        public void acquired() {
            ConcurrentEntry entry = (ConcurrentEntry)this._weak.get();
            if (entry == null) {
                return;
            }
            while (this._strong == null && !entry.isTerminated()) {
                Thread.onSpinWait();
            }
            this._strong = null;
        }

        public String toString() {
            return "%s@%x{%s,%s}".formatted(this.getClass().getSimpleName(), this.hashCode(), this._strong == null ? "acquired" : "released", this._weak.get());
        }
    }
}

