package com.paystack.android_sdk.ui.components.views.inputs

import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.text.BasicTextField
import androidx.compose.foundation.text.KeyboardActions
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.Immutable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.Shape
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.input.ImeAction
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.DpSize
import androidx.compose.ui.unit.dp
import com.paystack.android_sdk.ui.theme.pinInputFieldCellHeight
import com.paystack.android_sdk.ui.theme.pinInputFieldCellWidth
import com.paystack.android_sdk.ui.theme.pinInputUnfocusedBorderColor
import com.paystack.androiddesignsystem.PaystackTheme
import kotlinx.coroutines.delay

@Composable
internal fun PinInputField(
    text: String,
    onTextChanged: (String) -> Unit,
    modifier: Modifier = Modifier,
    pinLength: Int = DEFAULT_PIN_LENGTH,
    cellSpacing: Dp = PaystackTheme.dimensions.spacing.single,
    cellWidth: Dp = PaystackTheme.dimensions.pinInputFieldCellWidth,
    cellHeight: Dp = PaystackTheme.dimensions.pinInputFieldCellHeight,
    cellShape: Shape = RoundedCornerShape(PaystackTheme.dimensions.borderRadius.small),
    keyboardActions: KeyboardActions = KeyboardActions.Default,
    keyboardOptions: KeyboardOptions = KeyboardOptions(
        keyboardType = KeyboardType.NumberPassword,
        imeAction = ImeAction.Done
    ),
) {
    BasicTextField(
        value = text,
        onValueChange = onTextChanged,
        keyboardOptions = keyboardOptions,
        keyboardActions = keyboardActions,
        decorationBox = {
            Row(
                modifier = modifier,
                horizontalArrangement = Arrangement.spacedBy(cellSpacing)
            ) {
                val cellSize = DpSize(cellWidth, cellHeight)
                text.forEachIndexed { _, char ->
                    PinCharCell(
                        char = char,
                        isFocused = false,
                        shape = cellShape,
                        size = cellSize
                    )
                }

                repeat(pinLength - text.length) { index ->
                    PinCharCell(
                        char = ' ',
                        isFocused = index == 0,
                        shape = cellShape,
                        size = cellSize
                    )
                }
            }
        }
    )
}

@Composable
private fun PinCharBox(
    text: Char,
    isFocused: Boolean,
    textStyle: TextStyle,
    shape: Shape,
    size: DpSize,
    modifier: Modifier = Modifier,
    colors: PinInputFieldColors = pinInputFieldColors()
) {
    Box(
        modifier = modifier.size(size)
            .run {
                border(
                    width = 1.dp,
                    color = if (isFocused) colors.focusedContainerColor else colors.unfocusedContainerColor,
                    shape = shape,
                )
            },
        contentAlignment = Alignment.Center,
    ) {
        Text(
            text = text.toString(),
            modifier = Modifier,
            style = textStyle,
            color = if (isFocused) colors.focusedTextColor else colors.unfocusedTextColor,
        )
    }
}

@Composable
fun PinCharCell(
    char: Char,
    isFocused: Boolean,
    shape: Shape,
    size: DpSize,
    modifier: Modifier = Modifier,
    maskChar: Char = '•',
    colors: PinInputFieldColors = pinInputFieldColors()
) {
    var peek by remember { mutableStateOf(true) }
    LaunchedEffect(char) {
        if (char.isBlank()) {
            delay(CHAR_PEEK_DELAY_MS)
            peek = false
        }
    }

    val textStyle = if (peek) {
        PaystackTheme.typography.body.body14R
    } else {
        PaystackTheme.typography.heading.heading2
    }
    PinCharBox(
        text = if (peek) char else maskChar,
        isFocused = isFocused,
        textStyle = textStyle,
        shape = shape,
        modifier = modifier,
        colors = colors,
        size = size,
    )
}

@Immutable
class PinInputFieldColors internal constructor(
    val focusedTextColor: Color,
    val unfocusedTextColor: Color,
    val focusedContainerColor: Color,
    val unfocusedContainerColor: Color,
)

@Composable
fun pinInputFieldColors(
    focusedTextColor: Color = PaystackTheme.colors.primaryColors.stackBlue,
    unfocusedTextColor: Color = PaystackTheme.colors.primaryColors.stackBlue,
    focusedContainerColor: Color = PaystackTheme.colors.secondaryColors.stackGreen,
    unfocusedContainerColor: Color = PaystackTheme.colors.pinInputUnfocusedBorderColor,
) = PinInputFieldColors(
    focusedTextColor = focusedTextColor,
    unfocusedTextColor = unfocusedTextColor,
    focusedContainerColor = focusedContainerColor,
    unfocusedContainerColor = unfocusedContainerColor,
)

@Composable
@Preview
fun PinInputFieldPreview() {
    MaterialTheme {
        Surface {
            Box(
                modifier = Modifier.fillMaxSize(),
                contentAlignment = Alignment.Center
            ) {
                val (text, setText) = remember { mutableStateOf("") }
                PinInputField(text = text, onTextChanged = setText)
            }
        }
    }
}

private fun Char.isBlank() = this != ' '

const val DEFAULT_PIN_LENGTH = 4
const val CHAR_PEEK_DELAY_MS = 250L
