package com.paystack.android_sdk.ui.paymentchannels.card.auth.otp

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.style.TextAlign
import androidx.lifecycle.viewmodel.compose.viewModel
import com.paystack.android_sdk.ui.R
import com.paystack.android_sdk.ui.components.buttons.PrimaryButton
import com.paystack.android_sdk.ui.components.buttons.StackTextButton
import com.paystack.android_sdk.ui.components.views.inputs.PrimaryTextInputField
import com.paystack.android_sdk.ui.components.views.text.CountdownText
import com.paystack.android_sdk.ui.components.views.text.PaymentTitleText
import com.paystack.android_sdk.ui.models.Charge
import com.paystack.android_sdk.ui.paymentchannels.card.CardViewModelsProviderFactory
import com.paystack.androiddesignsystem.PaystackTheme

@Composable
internal fun OtpAuthForm(
    instruction: String,
    viewModelsProviderFactory: CardViewModelsProviderFactory,
    onCancel: () -> Unit,
    onResult: (Result<Charge>) -> Unit,
    modifier: Modifier = Modifier,
    viewModel: OtpFormViewModel = viewModel(factory = viewModelsProviderFactory)
) {
    val state by viewModel.state.collectAsState()
    val result by viewModel.result.collectAsState()

    LaunchedEffect(result) {
        result?.let { onResult(it) }
    }

    Column(
        modifier = modifier,
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        Image(painterResource(R.drawable.ic_otp), contentDescription = null)

        Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.triple))

        PaymentTitleText(
            titleText = instruction,
            modifier = Modifier.padding(horizontal = PaystackTheme.dimensions.spacing.triple)
        )

        Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))

        PrimaryTextInputField(
            modifier = Modifier.fillMaxWidth(),
            label = stringResource(R.string.otp),
            placeholder = stringResource(R.string.otp_input_placeholder),
            keyboardType = KeyboardType.NumberPassword,
            text = state.otp,
            onTextChanged = viewModel::onOtpChanged
        )

        Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))

        PrimaryButton(
            buttonText = stringResource(R.string.card_payment_authorize_payment),
            onClick = viewModel::submit,
            isLoading = state.isSubmitting,
            isEnabled = state.isValid
        )
        Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))

        AnimatedVisibility(visible = !state.isSubmitting) {
            if (state.otpResendLimited) {
                ResendOtpDisabled(onCancel)
            } else {
                ResendOtpButton(
                    enableResendOtp = state.enableResendOtp,
                    resendDelay = state.otpResendDelayMillis,
                    onResendClick = viewModel::requestResendOtp
                )
            }
        }
    }
}

@Composable
private fun ResendOtpDisabled(onCancel: () -> Unit) {
    Column(
        modifier = Modifier.padding(horizontal = PaystackTheme.dimensions.spacing.quad),
        horizontalAlignment = Alignment.CenterHorizontally,
    ) {
        Text(
            text = stringResource(R.string.resend_otp_disabled_message),
            textAlign = TextAlign.Center,
            style = PaystackTheme.typography.body.body14R,
            color = PaystackTheme.colors.primaryColors.navy03
        )

        Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))

        StackTextButton(
            text = stringResource(R.string.payment_action_cancel_text),
            onClick = onCancel
        )
    }
}

@Composable
private fun ResendOtpButton(
    enableResendOtp: Boolean,
    resendDelay: Long,
    onResendClick: () -> Unit,
) {
    StackTextButton(
        onClick = onResendClick,
        enabled = enableResendOtp
    ) {
        if (!enableResendOtp) {
            CountdownText(
                timeMillis = resendDelay,
                prefix = stringResource(R.string.otp_resend_countdown_prefix),
            )
        } else {
            Text(
                text = stringResource(R.string.resend_otp),
                color = PaystackTheme.colors.secondaryColors.stackGreen
            )
        }
    }
}
