package com.palantir.dialogue.hc4;

import com.codahale.metrics.Meter;
import com.palantir.logsafe.Preconditions;
import com.palantir.tritium.metrics.registry.MetricName;
import com.palantir.tritium.metrics.registry.TaggedMetricRegistry;
import java.util.Optional;

/** Dialogue client response metrics provided by the Apache client channel. */
final class DialogueClientMetrics {
    private static final String LIBRARY_NAME = "dialogue";

    private static final String LIBRARY_VERSION =
            Optional.ofNullable(DialogueClientMetrics.class.getPackage().getImplementationVersion())
                    .orElse("unknown");

    private final TaggedMetricRegistry registry;

    private DialogueClientMetrics(TaggedMetricRegistry registry) {
        this.registry = registry;
    }

    public static DialogueClientMetrics of(TaggedMetricRegistry registry) {
        return new DialogueClientMetrics(
                Preconditions.checkNotNull(registry, "TaggedMetricRegistry"));
    }

    /**
     * Rate that responses are garbage collected without being closed. This should only occur in the
     * case of a programming error.
     */
    ResponseLeakBuilderClientNameStage responseLeak() {
        return new ResponseLeakBuilder();
    }

    /** Marked every time a new client is created. */
    CreateBuilderClientNameStage create() {
        return new CreateBuilder();
    }

    /**
     * Marked every time an Apache client is successfully closed and any underlying resources
     * released (e.g. connections and background threads).
     */
    CloseBuilderClientNameStage close() {
        return new CloseBuilder();
    }

    @Override
    public String toString() {
        return "DialogueClientMetrics{registry=" + registry + '}';
    }

    interface ResponseLeakBuildStage {
        Meter build();
    }

    interface ResponseLeakBuilderClientNameStage {
        ResponseLeakBuilderServiceNameStage clientName(String clientName);
    }

    interface ResponseLeakBuilderServiceNameStage {
        ResponseLeakBuilderEndpointStage serviceName(String serviceName);
    }

    interface ResponseLeakBuilderEndpointStage {
        ResponseLeakBuildStage endpoint(String endpoint);
    }

    private final class ResponseLeakBuilder
            implements ResponseLeakBuilderClientNameStage,
                    ResponseLeakBuilderServiceNameStage,
                    ResponseLeakBuilderEndpointStage,
                    ResponseLeakBuildStage {
        private String clientName;

        private String serviceName;

        private String endpoint;

        @Override
        public Meter build() {
            return registry.meter(
                    MetricName.builder()
                            .safeName("dialogue.client.response.leak")
                            .putSafeTags("client-name", clientName)
                            .putSafeTags("service-name", serviceName)
                            .putSafeTags("endpoint", endpoint)
                            .putSafeTags("libraryName", LIBRARY_NAME)
                            .putSafeTags("libraryVersion", LIBRARY_VERSION)
                            .build());
        }

        @Override
        public ResponseLeakBuilder clientName(String clientName) {
            Preconditions.checkState(this.clientName == null, "client-name is already set");
            this.clientName = Preconditions.checkNotNull(clientName, "client-name is required");
            return this;
        }

        @Override
        public ResponseLeakBuilder serviceName(String serviceName) {
            Preconditions.checkState(this.serviceName == null, "service-name is already set");
            this.serviceName = Preconditions.checkNotNull(serviceName, "service-name is required");
            return this;
        }

        @Override
        public ResponseLeakBuilder endpoint(String endpoint) {
            Preconditions.checkState(this.endpoint == null, "endpoint is already set");
            this.endpoint = Preconditions.checkNotNull(endpoint, "endpoint is required");
            return this;
        }
    }

    interface CreateBuildStage {
        Meter build();
    }

    interface CreateBuilderClientNameStage {
        CreateBuilderClientTypeStage clientName(String clientName);
    }

    interface CreateBuilderClientTypeStage {
        CreateBuildStage clientType(String clientType);
    }

    private final class CreateBuilder
            implements CreateBuilderClientNameStage,
                    CreateBuilderClientTypeStage,
                    CreateBuildStage {
        private String clientName;

        private String clientType;

        @Override
        public Meter build() {
            return registry.meter(
                    MetricName.builder()
                            .safeName("dialogue.client.create")
                            .putSafeTags("client-name", clientName)
                            .putSafeTags("client-type", clientType)
                            .putSafeTags("libraryName", LIBRARY_NAME)
                            .putSafeTags("libraryVersion", LIBRARY_VERSION)
                            .build());
        }

        @Override
        public CreateBuilder clientName(String clientName) {
            Preconditions.checkState(this.clientName == null, "client-name is already set");
            this.clientName = Preconditions.checkNotNull(clientName, "client-name is required");
            return this;
        }

        @Override
        public CreateBuilder clientType(String clientType) {
            Preconditions.checkState(this.clientType == null, "client-type is already set");
            this.clientType = Preconditions.checkNotNull(clientType, "client-type is required");
            return this;
        }
    }

    interface CloseBuildStage {
        Meter build();
    }

    interface CloseBuilderClientNameStage {
        CloseBuilderClientTypeStage clientName(String clientName);
    }

    interface CloseBuilderClientTypeStage {
        CloseBuildStage clientType(String clientType);
    }

    private final class CloseBuilder
            implements CloseBuilderClientNameStage, CloseBuilderClientTypeStage, CloseBuildStage {
        private String clientName;

        private String clientType;

        @Override
        public Meter build() {
            return registry.meter(
                    MetricName.builder()
                            .safeName("dialogue.client.close")
                            .putSafeTags("client-name", clientName)
                            .putSafeTags("client-type", clientType)
                            .putSafeTags("libraryName", LIBRARY_NAME)
                            .putSafeTags("libraryVersion", LIBRARY_VERSION)
                            .build());
        }

        @Override
        public CloseBuilder clientName(String clientName) {
            Preconditions.checkState(this.clientName == null, "client-name is already set");
            this.clientName = Preconditions.checkNotNull(clientName, "client-name is required");
            return this;
        }

        @Override
        public CloseBuilder clientType(String clientType) {
            Preconditions.checkState(this.clientType == null, "client-type is already set");
            this.clientType = Preconditions.checkNotNull(clientType, "client-type is required");
            return this;
        }
    }
}
