package com.palantir.dialogue.hc4;

import com.codahale.metrics.Gauge;
import com.palantir.logsafe.Preconditions;
import com.palantir.tritium.metrics.registry.MetricName;
import com.palantir.tritium.metrics.registry.TaggedMetricRegistry;
import java.util.Optional;

/** Connection pool metrics from the dialogue Apache client. */
final class DialogueClientPoolMetrics {
    private static final String LIBRARY_NAME = "dialogue";

    private static final String LIBRARY_VERSION =
            Optional.ofNullable(
                            DialogueClientPoolMetrics.class.getPackage().getImplementationVersion())
                    .orElse("unknown");

    private final TaggedMetricRegistry registry;

    private DialogueClientPoolMetrics(TaggedMetricRegistry registry) {
        this.registry = registry;
    }

    public static DialogueClientPoolMetrics of(TaggedMetricRegistry registry) {
        return new DialogueClientPoolMetrics(
                Preconditions.checkNotNull(registry, "TaggedMetricRegistry"));
    }

    /**
     * Number of connections in the client connection pool in states <code>idle</code>, <code>
     * pending</code>, and <code>leased</code>.
     */
    SizeBuilderClientNameStage size() {
        return new SizeBuilder();
    }

    @Override
    public String toString() {
        return "DialogueClientPoolMetrics{registry=" + registry + '}';
    }

    interface SizeBuildStage {
        void build(Gauge<?> gauge);

        MetricName buildMetricName();
    }

    interface SizeBuilderClientNameStage {
        SizeBuilderStateStage clientName(String clientName);
    }

    interface SizeBuilderStateStage {
        SizeBuildStage state(String state);
    }

    private final class SizeBuilder
            implements SizeBuilderClientNameStage, SizeBuilderStateStage, SizeBuildStage {
        private String clientName;

        private String state;

        @Override
        public void build(Gauge<?> gauge) {
            registry.registerWithReplacement(buildMetricName(), gauge);
        }

        @Override
        public MetricName buildMetricName() {
            return MetricName.builder()
                    .safeName("dialogue.client.pool.size")
                    .putSafeTags("client-name", clientName)
                    .putSafeTags("state", state)
                    .putSafeTags("libraryName", LIBRARY_NAME)
                    .putSafeTags("libraryVersion", LIBRARY_VERSION)
                    .build();
        }

        @Override
        public SizeBuilder clientName(String clientName) {
            Preconditions.checkState(this.clientName == null, "client-name is already set");
            this.clientName = Preconditions.checkNotNull(clientName, "client-name is required");
            return this;
        }

        @Override
        public SizeBuilder state(String state) {
            Preconditions.checkState(this.state == null, "state is already set");
            this.state = Preconditions.checkNotNull(state, "state is required");
            return this;
        }
    }
}
