package com.palantir.dialogue.hc4;

import com.codahale.metrics.Meter;
import com.palantir.logsafe.Preconditions;
import com.palantir.tritium.metrics.registry.MetricName;
import com.palantir.tritium.metrics.registry.TaggedMetricRegistry;

/** Dialogue client response metrics provided by the Apache client channel. */
final class DialogueClientMetrics {
    private final TaggedMetricRegistry registry;

    private DialogueClientMetrics(TaggedMetricRegistry registry) {
        this.registry = registry;
    }

    public static DialogueClientMetrics of(TaggedMetricRegistry registry) {
        return new DialogueClientMetrics(
                Preconditions.checkNotNull(registry, "TaggedMetricRegistry"));
    }

    /**
     * Rate that responses are garbage collected without being closed. This should only occur in the
     * case of a programming error.
     */
    ResponseLeakBuilderClientNameStage responseLeak() {
        return new ResponseLeakBuilder();
    }

    @Override
    public String toString() {
        return "DialogueClientMetrics{registry=" + registry + '}';
    }

    interface ResponseLeakBuildStage {
        Meter build();
    }

    interface ResponseLeakBuilderClientNameStage {
        ResponseLeakBuilderServiceNameStage clientName(String clientName);
    }

    interface ResponseLeakBuilderServiceNameStage {
        ResponseLeakBuilderEndpointStage serviceName(String serviceName);
    }

    interface ResponseLeakBuilderEndpointStage {
        ResponseLeakBuildStage endpoint(String endpoint);
    }

    private final class ResponseLeakBuilder
            implements ResponseLeakBuilderClientNameStage,
                    ResponseLeakBuilderServiceNameStage,
                    ResponseLeakBuilderEndpointStage,
                    ResponseLeakBuildStage {
        private String clientName;

        private String serviceName;

        private String endpoint;

        @Override
        public Meter build() {
            return registry.meter(
                    MetricName.builder()
                            .safeName("dialogue.client.response.leak")
                            .putSafeTags("client-name", clientName)
                            .putSafeTags("service-name", serviceName)
                            .putSafeTags("endpoint", endpoint)
                            .build());
        }

        @Override
        public ResponseLeakBuilder clientName(String clientName) {
            Preconditions.checkState(this.clientName == null, "client-name is already set");
            this.clientName = Preconditions.checkNotNull(clientName, "client-name is required");
            return this;
        }

        @Override
        public ResponseLeakBuilder serviceName(String serviceName) {
            Preconditions.checkState(this.serviceName == null, "service-name is already set");
            this.serviceName = Preconditions.checkNotNull(serviceName, "service-name is required");
            return this;
        }

        @Override
        public ResponseLeakBuilder endpoint(String endpoint) {
            Preconditions.checkState(this.endpoint == null, "endpoint is already set");
            this.endpoint = Preconditions.checkNotNull(endpoint, "endpoint is required");
            return this;
        }
    }
}
