/*
 * Decompiled with CFR 0.152.
 */
package com.orbitz.consul.cache;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.orbitz.consul.ConsulException;
import com.orbitz.consul.async.ConsulResponseCallback;
import com.orbitz.consul.model.ConsulResponse;
import com.orbitz.consul.option.ImmutableQueryOptions;
import com.orbitz.consul.option.QueryOptions;
import java.math.BigInteger;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConsulCache<K, V> {
    private static final Logger LOGGER = LoggerFactory.getLogger(ConsulCache.class);
    @VisibleForTesting
    static final String BACKOFF_DELAY_PROPERTY = "com.orbitz.consul.cache.backOffDelay";
    private static final long BACKOFF_DELAY_QTY_IN_MS = ConsulCache.getBackOffDelayInMs(System.getProperties());
    private final AtomicReference<BigInteger> latestIndex = new AtomicReference<Object>(null);
    private final AtomicReference<ImmutableMap<K, V>> lastResponse = new AtomicReference<ImmutableMap>(ImmutableMap.of());
    private final AtomicReference<State> state = new AtomicReference<State>(State.latent);
    private final CountDownLatch initLatch = new CountDownLatch(1);
    private final ScheduledExecutorService executorService = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setDaemon(true).build());
    private final CopyOnWriteArrayList<Listener<K, V>> listeners = new CopyOnWriteArrayList();
    private final Function<V, K> keyConversion;
    private final CallbackConsumer<V> callBackConsumer;
    private final ConsulResponseCallback<List<V>> responseCallback;

    ConsulCache(Function<V, K> keyConversion, CallbackConsumer<V> callbackConsumer) {
        this.keyConversion = keyConversion;
        this.callBackConsumer = callbackConsumer;
        this.responseCallback = new ConsulResponseCallback<List<V>>(){

            @Override
            public void onComplete(ConsulResponse<List<V>> consulResponse) {
                if (consulResponse.isKnownLeader()) {
                    boolean changed;
                    if (!ConsulCache.this.isRunning()) {
                        return;
                    }
                    ConsulCache.this.updateIndex(consulResponse);
                    LOGGER.debug("Consul cache updated (index={})", (Object)ConsulCache.this.latestIndex);
                    ImmutableMap full = ConsulCache.this.convertToMap(consulResponse);
                    boolean bl = changed = !full.equals(ConsulCache.this.lastResponse.get());
                    if (changed) {
                        ConsulCache.this.lastResponse.set(full);
                    }
                    if (changed) {
                        for (Listener l : ConsulCache.this.listeners) {
                            l.notify(full);
                        }
                    }
                    if (ConsulCache.this.state.compareAndSet(State.starting, State.started)) {
                        ConsulCache.this.initLatch.countDown();
                    }
                    ConsulCache.this.runCallback();
                } else {
                    this.onFailure(new ConsulException("Consul cluster has no elected leader"));
                }
            }

            @Override
            public void onFailure(Throwable throwable) {
                if (!ConsulCache.this.isRunning()) {
                    return;
                }
                LOGGER.error(String.format("Error getting response from consul. will retry in %d %s", new Object[]{BACKOFF_DELAY_QTY_IN_MS, TimeUnit.MILLISECONDS}), throwable);
                ConsulCache.this.executorService.schedule(new Runnable(){

                    @Override
                    public void run() {
                        ConsulCache.this.runCallback();
                    }
                }, BACKOFF_DELAY_QTY_IN_MS, TimeUnit.MILLISECONDS);
            }
        };
    }

    @VisibleForTesting
    static long getBackOffDelayInMs(Properties properties) {
        String backOffDelay = null;
        try {
            backOffDelay = properties.getProperty(BACKOFF_DELAY_PROPERTY);
            if (!Strings.isNullOrEmpty((String)backOffDelay)) {
                return Long.parseLong(backOffDelay);
            }
        }
        catch (Exception ex) {
            LOGGER.warn(backOffDelay != null ? String.format("Error parsing property variable %s: %s", BACKOFF_DELAY_PROPERTY, backOffDelay) : String.format("Error extracting property variable %s", BACKOFF_DELAY_PROPERTY), (Throwable)ex);
        }
        return TimeUnit.SECONDS.toMillis(10L);
    }

    public void start() throws Exception {
        Preconditions.checkState((boolean)this.state.compareAndSet(State.latent, State.starting), (String)"Cannot transition from state %s to %s", (Object[])new Object[]{this.state.get(), State.starting});
        this.runCallback();
    }

    public void stop() throws Exception {
        State previous = this.state.getAndSet(State.stopped);
        if (previous != State.stopped) {
            this.executorService.shutdownNow();
        }
    }

    private void runCallback() {
        if (this.isRunning()) {
            this.callBackConsumer.consume(this.latestIndex.get(), this.responseCallback);
        }
    }

    private boolean isRunning() {
        return this.state.get() == State.started || this.state.get() == State.starting;
    }

    public boolean awaitInitialized(long timeout, TimeUnit unit) throws InterruptedException {
        return this.initLatch.await(timeout, unit);
    }

    public ImmutableMap<K, V> getMap() {
        return this.lastResponse.get();
    }

    @VisibleForTesting
    ImmutableMap<K, V> convertToMap(ConsulResponse<List<V>> response) {
        if (response == null || response.getResponse() == null || response.getResponse().isEmpty()) {
            return ImmutableMap.of();
        }
        ImmutableMap.Builder builder = ImmutableMap.builder();
        HashSet<Object> keySet = new HashSet<Object>();
        for (V v : response.getResponse()) {
            Object key = this.keyConversion.apply(v);
            if (key != null) {
                if (!keySet.contains(key)) {
                    builder.put(key, v);
                } else {
                    LOGGER.warn("Duplicate service encountered. May differ by tags. Try using more specific tags? " + key.toString());
                }
            }
            keySet.add(key);
        }
        return builder.build();
    }

    private void updateIndex(ConsulResponse<List<V>> consulResponse) {
        if (consulResponse != null && consulResponse.getIndex() != null) {
            this.latestIndex.set(consulResponse.getIndex());
        }
    }

    protected static QueryOptions watchParams(BigInteger index, int blockSeconds, QueryOptions queryOptions) {
        Preconditions.checkArgument((!queryOptions.getIndex().isPresent() && !queryOptions.getWait().isPresent() ? 1 : 0) != 0, (Object)"Index and wait cannot be overridden");
        return ImmutableQueryOptions.builder().from(ConsulCache.watchDefaultParams(index, blockSeconds)).token(queryOptions.getToken()).consistencyMode(queryOptions.getConsistencyMode()).near(queryOptions.getNear()).datacenter(queryOptions.getDatacenter()).build();
    }

    private static QueryOptions watchDefaultParams(BigInteger index, int blockSeconds) {
        if (index == null) {
            return QueryOptions.BLANK;
        }
        return QueryOptions.blockSeconds(blockSeconds, index).build();
    }

    public boolean addListener(Listener<K, V> listener) {
        boolean added = this.listeners.add(listener);
        if (this.state.get() == State.started) {
            listener.notify((Map)this.lastResponse.get());
        }
        return added;
    }

    public List<Listener<K, V>> getListeners() {
        return Collections.unmodifiableList(this.listeners);
    }

    public boolean removeListener(Listener<K, V> listener) {
        return this.listeners.remove(listener);
    }

    @VisibleForTesting
    protected State getState() {
        return this.state.get();
    }

    public static interface Listener<K, V> {
        public void notify(Map<K, V> var1);
    }

    protected static interface CallbackConsumer<V> {
        public void consume(BigInteger var1, ConsulResponseCallback<List<V>> var2);
    }

    static enum State {
        latent,
        starting,
        started,
        stopped;

    }
}

