package com.onradar.sdk.model;

import org.json.JSONException;
import org.json.JSONObject;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

public class RadarEvent {

    public enum RadarEventType {
        USER_ENTERED_GEOFENCE, USER_EXITED_GEOFENCE
    }

    private String mId;
    private Date mCreatedAt;
    private boolean mLive;
    private RadarEventType mType;
    private RadarGeofence mGeofence;
    private float mDuration;

    public RadarEvent(String _id, Date createdAt, boolean live, RadarEventType type, RadarGeofence geofence, float duration) {
        mId = _id;
        mCreatedAt = createdAt;
        mLive = live;
        mType = type;
        mGeofence = geofence;
        mDuration = duration;
    }

    public RadarEvent(JSONObject obj) throws JSONException, ParseException {
        if (obj.has("_id"))
            mId = obj.getString("_id");

        if (obj.has("createdAt")) {
            String eventCreatedAtStr = obj.getString("createdAt");
            SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
            mCreatedAt = dateFormat.parse(eventCreatedAtStr);
        }

        if (obj.has("live"))
            mLive = obj.getBoolean("live");

        if (obj.has("type")) {
            String typeStr = obj.getString("type");
            if (typeStr.equals("user.entered_geofence"))
                mType = RadarEventType.USER_ENTERED_GEOFENCE;
            else if (typeStr.equals("user.exited_geofence"))
                mType = RadarEventType.USER_EXITED_GEOFENCE;
        }

        if (obj.has("geofence")) {
            JSONObject geofenceObj = obj.getJSONObject("geofence");
            mGeofence = new RadarGeofence(geofenceObj);
        }

        if (obj.has("duration"))
            mDuration = (float)obj.getDouble("duration");
    }

    /**
     * @return The unique ID for the event, provided by Radar.
     */
    public String getId() {
        return mId;
    }

    /**
     * @return The datetime when the event was created.
     */
    public Date getCreatedAt() {
        return mCreatedAt;
    }

    /**
     * @return A boolean indicating whether the event was generated for a user created with your live API key.
     */
    public boolean getLive() {
        return mLive;
    }

    /**
     * @return The type of event.
     */
    public RadarEventType getType() {
        return mType;
    }

    /**
     * @return The geofence for which the event was generated.
     */
    public RadarGeofence getGeofence() {
        return mGeofence;
    }

    /**
     * @return The duration between entry and exit events, in minutes, for exit events. 0 for entry events.
     */
    public float getDuration() {
        return mDuration;
    }

}
