package com.onradar.sdk.model;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;

public class RadarEvent {

    public enum RadarEventType {
        UNKNOWN, USER_ENTERED_GEOFENCE, USER_EXITED_GEOFENCE, USER_ENTERED_HOME, USER_EXITED_HOME, USER_ENTERED_OFFICE, USER_EXITED_OFFICE, USER_STARTED_TRAVELING, USER_STOPPED_TRAVELING
    }

    public enum RadarEventConfidence {
        NONE, LOW, MEDIUM, HIGH
    }

    private String mId;
    private Date mCreatedAt;
    private boolean mLive;
    private RadarEventType mType;
    private RadarGeofence mGeofence;
    private RadarEventConfidence mConfidence;
    private float mDuration;

    public static RadarEvent[] eventsFromJSONArray(JSONArray array) throws JSONException, ParseException {
        int numEvents = array.length();
        RadarEvent[] events = new RadarEvent[numEvents];
        for (int i = 0; i < numEvents; i++) {
            JSONObject eventObj = array.getJSONObject(i);
            RadarEvent event = new RadarEvent(eventObj);
            events[i] = event;
        }
        return events;
    }

    public RadarEvent(String _id, Date createdAt, boolean live, RadarEventType type, RadarGeofence geofence, RadarEventConfidence confidence, float duration) {
        mId = _id;
        mCreatedAt = createdAt;
        mLive = live;
        mType = type;
        mGeofence = geofence;
        mConfidence = confidence;
        mDuration = duration;
    }

    public RadarEvent(JSONObject obj) throws JSONException, ParseException {
        if (obj.has("_id"))
            mId = obj.getString("_id");

        if (obj.has("createdAt")) {
            String createdAtStr = obj.getString("createdAt");
            DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
            dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
            mCreatedAt = dateFormat.parse(createdAtStr);
        }

        if (obj.has("live"))
            mLive = obj.getBoolean("live");

        if (obj.has("type")) {
            String typeStr = obj.getString("type");
            if (typeStr.equals("user.entered_geofence"))
                mType = RadarEventType.USER_ENTERED_GEOFENCE;
            else if (typeStr.equals("user.exited_geofence"))
                mType = RadarEventType.USER_EXITED_GEOFENCE;
            else if (typeStr.equals("user.entered_home"))
                mType = RadarEventType.USER_ENTERED_HOME;
            else if (typeStr.equals("user.exited_home"))
                mType = RadarEventType.USER_EXITED_HOME;
            else if (typeStr.equals("user.entered_office"))
                mType = RadarEventType.USER_ENTERED_OFFICE;
            else if (typeStr.equals("user.exited_office"))
                mType = RadarEventType.USER_EXITED_OFFICE;
            else if (typeStr.equals("user.started_traveling"))
                mType = RadarEventType.USER_STARTED_TRAVELING;
            else if (typeStr.equals("user.stopped_traveling"))
                mType = RadarEventType.USER_STOPPED_TRAVELING;
            else
                mType = RadarEventType.UNKNOWN;
        }

        if (obj.has("geofence")) {
            JSONObject geofenceObj = obj.getJSONObject("geofence");
            mGeofence = new RadarGeofence(geofenceObj);
        }

        if (obj.has("confidence")) {
            int confidenceInt = obj.getInt("confidence");
            if (confidenceInt == 3)
                mConfidence = RadarEventConfidence.HIGH;
            else if (confidenceInt == 2)
                mConfidence = RadarEventConfidence.MEDIUM;
            else if (confidenceInt == 1)
                mConfidence = RadarEventConfidence.LOW;
            else
                mConfidence = RadarEventConfidence.NONE;
        }

        if (obj.has("duration"))
            mDuration = (float)obj.getDouble("duration");
    }

    /**
     * @return The unique ID for the event, provided by Radar.
     */
    public String getId() {
        return mId;
    }

    /**
     * @return The datetime when the event was created.
     */
    public Date getCreatedAt() {
        return mCreatedAt;
    }

    /**
     * @return A boolean indicating whether the event was generated for a user created with your live API key.
     */
    public boolean getLive() {
        return mLive;
    }

    /**
     * @return The type of event.
     */
    public RadarEventType getType() {
        return mType;
    }

    /**
     * @return The geofence for which the event was generated. May be null for non-geofence events.
     */
    public RadarGeofence getGeofence() {
        return mGeofence;
    }

    /**
     * @return The confidence level of the event.
     */
    public RadarEventConfidence getConfidence() {
        return mConfidence;
    }

    /**
     * @return The duration between entry and exit events, in minutes, for exit events. 0 for entry events.
     */
    public float getDuration() {
        return mDuration;
    }

}
