// Subpackaged to prevent conflicts with other plugins
// version 1.1.4 code base
package com.onesignal.shortcutbadger;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.os.Build;
import android.util.Log;

import com.onesignal.shortcutbadger.impl.*;

import java.util.LinkedList;
import java.util.List;


/**
 * @author Leo Lin
 */
public final class ShortcutBadger {

   private static final String LOG_TAG = "ShortcutBadger";

   private static final List<Class<? extends Badger>> BADGERS = new LinkedList<Class<? extends Badger>>();

   static {
      BADGERS.add(AdwHomeBadger.class);
      BADGERS.add(ApexHomeBadger.class);
      BADGERS.add(NewHtcHomeBadger.class);
      BADGERS.add(NovaHomeBadger.class);
      BADGERS.add(SonyHomeBadger.class);
      BADGERS.add(XiaomiHomeBadger.class);
      BADGERS.add(AsusHomeLauncher.class);
      BADGERS.add(HuaweiHomeBadger.class);
   }

   private static Badger sShortcutBadger;
   private static ComponentName sComponentName;

   /**
    * Tries to update the notification count
    *
    * @param context    Caller context
    * @param badgeCount Desired badge count
    * @return true in case of success, false otherwise
    */
   public static boolean applyCount(Context context, int badgeCount) {
      try {
         applyCountOrThrow(context, badgeCount);
         return true;
      } catch (ShortcutBadgeException e) {
         Log.e(LOG_TAG, "Unable to execute badge", e);
         return false;
      }
   }

   /**
    * Tries to update the notification count, throw a {@link ShortcutBadgeException} if it fails
    *
    * @param context    Caller context
    * @param badgeCount Desired badge count
    */
   public static void applyCountOrThrow(Context context, int badgeCount) throws ShortcutBadgeException {
      if (sShortcutBadger == null) {
         boolean launcherReady = initBadger(context);

         if (!launcherReady)
            throw new ShortcutBadgeException("No default launcher available");
      }

      try {
         sShortcutBadger.executeBadge(context, sComponentName, badgeCount);
      } catch (Exception e) {
         throw new ShortcutBadgeException("Unable to execute badge", e);
      }
   }

   /**
    * Tries to remove the notification count
    *
    * @param context Caller context
    * @return true in case of success, false otherwise
    */
   public static boolean removeCount(Context context) {
      return applyCount(context, 0);
   }

   /**
    * Tries to remove the notification count, throw a {@link ShortcutBadgeException} if it fails
    *
    * @param context Caller context
    */
   public static void removeCountOrThrow(Context context) throws ShortcutBadgeException {
      applyCountOrThrow(context, 0);
   }

   // Initialize Badger if a launcher is availalble (eg. set as default on the device)
   // Returns true if a launcher is available, in this case, the Badger will be set and sShortcutBadger will be non null.
   private static boolean initBadger(Context context) {

      sComponentName = context.getPackageManager().getLaunchIntentForPackage(context.getPackageName()).getComponent();

      Intent intent = new Intent(Intent.ACTION_MAIN);
      intent.addCategory(Intent.CATEGORY_HOME);
      ResolveInfo resolveInfo = context.getPackageManager().resolveActivity(intent, PackageManager.MATCH_DEFAULT_ONLY);

      if (resolveInfo == null || resolveInfo.activityInfo.name.toLowerCase().contains("resolver"))
         return false;

      String currentHomePackage = resolveInfo.activityInfo.packageName;

      for (Class<? extends Badger> badger : BADGERS) {
         Badger shortcutBadger = null;
         try {
            shortcutBadger = badger.newInstance();
         } catch (Exception e) {
         }
         if (shortcutBadger != null && shortcutBadger.getSupportLaunchers().contains(currentHomePackage)) {
            sShortcutBadger = shortcutBadger;
            break;
         }
      }

      if (sShortcutBadger == null) {
         if (Build.MANUFACTURER.equalsIgnoreCase("Xiaomi"))
            sShortcutBadger = new XiaomiHomeBadger();
         else
            sShortcutBadger = new DefaultBadger();
      }

      return true;
   }

   // Avoid anybody to instantiate this class
   private ShortcutBadger() {

   }
}
