package com.noisyz.bindlibrary.base.impl.adapter;

import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import com.noisyz.bindlibrary.base.impl.ObjectViewBinder;
import com.noisyz.bindlibrary.callback.clickevent.OnElementClickListener;
import com.noisyz.bindlibrary.callback.clickevent.OnElementClickListenerWrapper;
import com.noisyz.bindlibrary.callback.clickevent.OnItemClickListener;
import com.noisyz.bindlibrary.callback.filter.ItemFilterCallback;
import com.noisyz.bindlibrary.callback.filter.ItemsFilter;
import com.noisyz.bindlibrary.callback.layoutcallback.EmptyBaseLayoutResourceProvider;
import com.noisyz.bindlibrary.callback.layoutcallback.LayoutResourceProvider;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by oleg on 06.09.16.
 */
public abstract class RecyclerBindAdapter<T> extends RecyclerView.Adapter<RecyclerView.ViewHolder> implements ItemsFilter {

    private List<T> itemList, filteredList;
    private LayoutResourceProvider<T> layoutResourceProvider;
    private OnItemClickListener<T> onItemClickListener;
    private List<OnElementClickListenerWrapper<T>> elementsClickWrappers;
    private ItemFilterCallback<T, Object> itemFilterCallback;

    public RecyclerBindAdapter(List<T> itemList, int layoutResID) {
        layoutResourceProvider = new EmptyBaseLayoutResourceProvider<>(layoutResID);
        if (itemList == null) {
            itemList = new ArrayList<>();
        }
        this.itemList = itemList;
        elementsClickWrappers = new ArrayList<>();
    }

    public RecyclerBindAdapter(List<T> itemList, LayoutResourceProvider<T> mLayoutResourceProvider) {
        this(mLayoutResourceProvider);
        if (itemList == null) {
            itemList = new ArrayList<>();
        }
        this.itemList = itemList;
    }

    private RecyclerBindAdapter(final LayoutResourceProvider<T> mLayoutResourceProvider) {
        layoutResourceProvider = mLayoutResourceProvider;
        elementsClickWrappers = new ArrayList<>();
    }

    public RecyclerBindAdapter setOnElementClickListener(int elementId, OnElementClickListener<T> onClickListener) {
        elementsClickWrappers.add(new OnElementClickListenerWrapper<>(elementId, onClickListener));
        return this;
    }

    public RecyclerBindAdapter setOnElementsClickListener(OnElementClickListener<T> onClickListener, int... elementIds) {
        elementsClickWrappers.add(new OnElementClickListenerWrapper<>(elementIds, onClickListener));
        return this;
    }

    public RecyclerBindAdapter setOnItemClickListener(OnItemClickListener<T> onItemClickListener) {
        this.onItemClickListener = onItemClickListener;
        return this;
    }

    public RecyclerBindAdapter setItemFilterCallback(ItemFilterCallback<T, Object> itemFilterCallback) {
        this.itemFilterCallback = itemFilterCallback;
        return this;
    }

    @Override
    public BindViewHolder onCreateViewHolder(ViewGroup parent, int itemViewType) {
        LayoutInflater inflater = LayoutInflater.from(parent.getContext());
        return new BindViewHolder(inflater.inflate(
                layoutResourceProvider.getLayoutResourceID(itemViewType), parent, false));
    }

    @Override
    public int getItemViewType(int position) {
        return layoutResourceProvider.getItemViewType(getItem(position));
    }


    @Override
    public void onBindViewHolder(RecyclerView.ViewHolder holder, final int position) {
        T t = getItem(position);
        BindViewHolder bindViewHolder = (BindViewHolder) holder;
        bindViewHolder.bindView(t);

        final View finalView = bindViewHolder.itemView;
        if (onItemClickListener != null) {
            finalView.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    onItemClickListener.onItemClick(finalView, position, getItem(position));
                }
            });
        }

        if (elementsClickWrappers != null && !elementsClickWrappers.isEmpty()) {
            for (final OnElementClickListenerWrapper<T> wrapper : elementsClickWrappers)
                for (final int elementId : wrapper.getIds()) {
                    View element = finalView.findViewById(elementId);
                    if (element != null) {
                        element.setOnClickListener(new View.OnClickListener() {
                            @Override
                            public void onClick(View v) {
                                wrapper.getOnClickListener().onElementClick(finalView, elementId, position, getItem(position));
                            }
                        });
                    }
                }
        }
    }

    @Override
    public int getItemCount() {
        List<T> list = itemFilterCallback != null ? filteredList : itemList;
        return list != null ? list.size() : 0;
    }

    public T getItem(int position) {
        List<T> list = itemFilterCallback != null ? filteredList : itemList;
        return list != null ? list.get(position) : null;
    }

    public List<T> getItems() {
        return itemList;
    }

    public void setItems(List<T> items) {
        this.itemList = items;
        notifyDataSetChanged();
    }

    public void filter(Object object) {
        if (itemFilterCallback != null) {
            filteredList = new ArrayList<>();
            for (T t : itemList) {
                if (itemFilterCallback.filterBy(t, object))
                    filteredList.add(t);
            }
            notifyDataSetChanged();
        }
    }

    protected abstract ObjectViewBinder<T> initObjectViewBinder(T t);

    private class BindViewHolder extends RecyclerView.ViewHolder {

        private ObjectViewBinder<T> objectViewBinder;

        public BindViewHolder(View itemView) {
            super(itemView);
        }

        public void bindView(T t) {
            if (objectViewBinder == null) {
                objectViewBinder = initObjectViewBinder(t).registerView(itemView);
            } else {
                objectViewBinder.registerView(itemView).setBindObject(t);
            }
            objectViewBinder.bindUI();
        }
    }
}
